#! /usr/bin/env perl

=head1 DESCRIPTION

Contains the functions which do the variable replacements

=head1 FUNCTIONS

=cut

package vr;  # variable replacer

use Dancer ':syntax';

=head2 dancer_dancer_param($ref)

Avoid error message for functions called outside a Dancer application

=cut
sub dancer_param { Dancer::SharedData->request ? param(shift) : undef }

=head2 %hash = fillDatabaseParams()

Extracts database params from config, then replace variable names by values from user parameters

=cut
sub fillDatabaseParams {
	my %spec = %{config->{database}}; foreach my $val (values (%spec)) { 
		unless (ref ($val)) { $val = interpolation($val); }
		else {
			foreach my $cond (@$val) {
				if ($cond->{type} eq 'default') { $val = $cond->{value}; last; }  

				my $expr = $cond->{expression}; $expr =~ s/\$(\w+)/dancer_param($1)/eg; 
				if (($cond->{type} eq 'regex') and ($expr =~ $cond->{expected})) { $val = $cond->{value}; last; }  
				if (($cond->{type} eq 'eq') and ($expr eq $cond->{expected})) { $val = $cond->{value}; last; }  
			}
		}
	}
	return %spec;
}

=head2 $v = interpolation($expr)

Translate $expr by replacing $xxx to dancer_param(xxx),
${xxx:0,2} by dancer_param(xxx).substring(0,2)

=cut
sub interpolation {
	my $expr = shift; my $vars = shift || {};
	if (ref ($expr) =~ /ARRAY/) {
		foreach my $cond (@$expr) {
			if ($cond->{type} eq 'default') { return interpolation($cond->{value}, $vars); }  

			my $expr = $cond->{expression}; $expr =~ s/\$(\w+)/$vars->{$1}/eg; 
			if (($cond->{type} eq 'regex') and ($expr =~ $cond->{expected})) { return interpolation($cond->{value}, $vars); }  
			if (($cond->{type} eq 'eq') and ($expr eq $cond->{expected})) {return interpolation($cond->{value}, $vars); }  
		}
	}
	return '' unless $expr; # avoid lot of warnings
	$expr =~ s/\$(\w+)/$vars->{$1} || dancer_param($1) || config->{alias}{$1}/eg;
	$expr =~ s/\$\{(\w+)\}/$vars->{$1} || dancer_param($1) || config->{alias}{$1}/eg;
	$expr =~ s/\$\{(\w+):(\d+),(\d+)\}/substr($vars->{$1} || dancer_param($1) || config->{alias}{$1},$2,$3)/eg;
	$expr =~ s/\$\{(\w+):(\d+)\+\}/substr($vars->{$1} || dancer_param($1) || config->{alias}{$1},$2)/eg;
	return $expr;
}

=head2 @tab = possibleValues($expr)

Generates an array where each variable present in $expr is replaced by all possible values

=cut
sub possibleValues {
	my $expr = shift;
	if (ref($expr)) {  # conditionnal
		my @all = ();
		foreach my $cond (@$expr) {
			push (@all, possibleValues($cond->{value}));
		}
		return @all;
	} elsif ($expr =~ /\$(\w+)/) {
		my $varName = $1; my @all = ();
		if (dancer_param($varName)) {
			my $copy = $expr; $copy =~ s/\$$varName/dancer_param($varName)/ge;
			my @list = possibleValues($copy); foreach my $item (@list) { $item->{$varName} = dancer_param($varName); }
			push (@all, @list);
		} else {
			foreach my $val (@{config->{'possible-values'}{$varName}}) { 
				my $copy = $expr; $copy =~ s/\$$varName/$val/ge;
				my @list = possibleValues($copy); foreach my $item (@list) { $item->{$varName} = $val; }
				push (@all, @list);
			}
		}
		return @all;
	} else {
		return ( {value => $expr } );
	}
}

use DBI;

=head2 @tab = memoriesList()

Create a list of objects, one for each existing memory. 

If filters are given in config, if the user gave a parameter value for them only the memories with given value are returned.
Else, all memories are given ordered by the filter values.

=cut
sub memoriesList {
        my $tableImplicitFilter = config->{database}{table}; $tableImplicitFilter = "public.$tableImplicitFilter" unless $tableImplicitFilter =~ /\./;
        $tableImplicitFilter =~ s!\$\{?(\w+)\}?!
          my @vals = possibleValues('$' . $1); @vals ? ('(' . join('|',map { $_->{value} } @vals) . ')') : '(\w+?)'
        !gex;
         $tableImplicitFilter = qr/^$tableImplicitFilter$/;

	my %MEM = ();
	foreach my $host (possibleValues (config->{database}{host})) {
		foreach my $port (possibleValues (config->{database}{port})) {
			foreach my $db (possibleValues (config->{database}{catalog})) {
				my %current = (%$host, %$port, %$db, host => $host->{value}, port => $port->{value}, catalog => $db->{value}); delete $current{value};
				my $DBI = DBI->connect("dbi:Pg:host=$host->{value};port=$port->{value};dbname=$db->{value}", config->{database}{user}, config->{database}{pass});
				next unless $DBI; # do not fail when connection does not work
				
				my $list_view = config->{list}{sql}{view}; $list_view = $list_view->{name} if ref $list_view; $list_view ||= 'information_schema.tables';
				my $st = $DBI->prepare ("select * from $list_view"); $st->execute;
			SQL_VIEW:
				while (my $hash = $st->fetchrow_hashref) {
					next if $hash->{table_schema} =~ /information_schema|pg_catalog/; # meta views, cannot be memories
					my %copy = (%current, %$hash); 
					if (ref (config->{list}{sql}{view}{fields}) =~ /HASH/) {
						while (my ($k, $v) = each (%{config->{list}{sql}{view}{fields}})) { $copy{$k} = interpolation($v, \%copy); }
					}
					if (config->{'filter-tables'}) { # explicit filter
						FILTER_TABLES:
						foreach my $filter (@{ config->{'filter-tables'} }) {
							next SQL_VIEW if $filter->{table_schema} and not ref($filter->{table_schema}) and $copy{table_schema} ne $filter->{table_schema};
						}
					}
					next SQL_VIEW unless "$copy{table_schema}.$copy{table_name}" =~ $tableImplicitFilter;
					my $DestTab = \%MEM; my $doLastUse = config->{list}{sql}{view}{'opt-fields'}{'last-use'};
					if (config->{'filter-parameters'}) {
						for (my $i = 0; $i < @{config->{'filter-parameters'}}; $i++) {
							my $var = ${config->{'filter-parameters'}}[$i];
							$copy{$var->{'var-name'}} ||= interpolation($var->{value}, \%copy);
							if (dancer_param($var->{'var-name'})) {
								next SQL_VIEW if $copy{$var->{'var-name'}} and (dancer_param($var->{'var-name'}) ne $copy{$var->{'var-name'}});	# Apply filter, if exists
							} elsif ($doLastUse eq 'filter') {
								$doLastUse = 0;
							}
							unless ($DestTab->{$copy{$var->{'var-name'}}}) {
								if ($i < @{config->{'filter-parameters'}} - 1) { 	
									$DestTab = $DestTab->{$copy{$var->{'var-name'}}} = {};	# Not last: create empty hash
								} else {
									$DestTab = $DestTab->{$copy{$var->{'var-name'}}} = [];  # Last : create empty table
								}
							} else {
								$DestTab = $DestTab->{$copy{$var->{'var-name'}}};	# Get already existing hash/table
							}
						}
					} else {
						$DestTab = $MEM{__DEFAULT__} ||= [];
					}
					calcGroupFields (\%copy, $doLastUse, $DBI);
					push (@$DestTab, \%copy);	# unless we called next SQL_VIEW
				}
				$DBI->disconnect;
			}
		}
	}
	return %MEM;
}

sub findMemory {
	my $spec = shift;
LOOP:
	foreach my $host (possibleValues (config->{database}{host})) {
		next LOOP unless lc($spec->{host}) eq lc($host->{value});
		foreach my $port (possibleValues (config->{database}{port})) {
			next LOOP unless lc($spec->{port}) eq $port->{value};
			foreach my $db (possibleValues (config->{database}{catalog})) {
				next LOOP unless $spec->{catalog} eq $db->{value};	# case sensitive
				my %current = (%$host, %$port, %$db, host => $host->{value}, port => $port->{value}, db => $db->{value}); delete $current{value};
				my $DBI = DBI->connect("dbi:Pg:host=$host->{value};port=$port->{value};dbname=$db->{value}", config->{database}{user}, config->{database}{pass});
				if ($DBI) {  # if not, does not matter
					my $list_view = config->{list}{sql}{view}; $list_view = $list_view->{name} if ref $list_view; $list_view ||= 'information_schema.tables';
					my $st = $DBI->prepare ("select * from $list_view"); $st->execute;
					while (my $hash = $st->fetchrow_hashref) {
						my %copy = (%current, %$hash);
						my ($schema, $table) = map { lc($_) } split(/\./, $spec->{table}); ($schema,$table) = ('public',$schema) unless $table;						 
						if (($schema eq $copy{table_schema}) and ($table eq $copy{table_name})) {
							calcGroupFields (\%copy, 1, $DBI);	
							1 while $st->fetchrow_hashref;
							if (shift) { $copy{DBI} = $DBI; } else { $DBI->disconnect; } 
							return \%copy;
						}
					}
					$DBI->disconnect; # next loop will use another one
				}
			}
		}
	}
	return undef;
}

my %existFields = ();

sub calcGroupFields {
	my $mem = shift; my $doAll = shift;
	my $DBI = shift; my $DBI_DEL; my @fields; my %pos;
	my ($schema, $tableName) = ($mem->{table_schema}, $mem->{table_name}); $schema ||= 'public';
FIELDS_LOOP:
	while (my ($name, $field) = each (%{config->{list}{sql}{view}{'opt-fields'}})) {
		next unless $field->{'group-function'};
		next if $field->{'if'} eq 'filter' and (! $doAll);
		# Now we really want this field
		$DBI ||= $DBI_DEL = DBI->connect(dbiSpec($mem), config->{database}{user}, config->{database}{pass}); 
		my $expr = $field->{'group-function'};
		if (ref ($field->{'db-field'})) {
			$expr .= "(coalesce(" . join(',',@{$field->{'db-field'}}) . "))";
			if ($field->{'check-exists'}) {
				foreach my $f0 (@{$field->{'check-exists'}}) {
					unless (defined $existFields{"dbi:Pg:host=$mem->{host};port=$mem->{port};dbname=$mem->{catalog};table=$schema.$tableName"}{$f0}) {
						# not yet known, check the tableName
						my $exist = $DBI->selectrow_array("SELECT column_name FROM information_schema.columns 
						   WHERE table_name='$tableName' and column_name='$f0' and table_schema='$schema'");
						$existFields{"dbi:Pg:host=$mem->{host};port=$mem->{port};dbname=$mem->{catalog};table=$schema.$tableName"}{$f0} = $exist || 0;					
					}
					$expr =~ s/$f0,// if $existFields{"dbi:Pg:host=$mem->{host};port=$mem->{port};dbname=$mem->{catalog};table=$schema.$tableName"}{$f0} == 0;
				}
				$expr =~ s/coalesce\((\w+)\)/$1/;
			}
		} elsif ($field->{'db-field'}) {
			$expr .= "($field->{'db-field'})";
		} else {
			$expr .= '(*)';
		}
		if ($expr =~ /^join/) {
			$expr = substr($expr, index($expr,'(') + 1); $expr =~ s/\s*\)$//;
			my $req = $DBI->prepare("select distinct $expr from $schema.$tableName"); $req->execute;
			my @res = (); while (my @t = $req->fetchrow_array) { push (@res,$t[0]); }
			if ($field->{'max-length'}) {			
				$mem->{$name} = '';
			RES_LOOP:
				while (my $x = shift(@res)) {
					if (length($mem->{$name}) < $field->{'max-length'}) { $mem->{$name} .= "$x,"; }
					else { $mem->{$name} .= "..."; last RES_LOOP; }
				}
				$mem->{$name} =~ s/,$//;
			} else {
				$mem->{$name} = join(',',@res);
			}
			next FIELDS_LOOP; # This is not a group field
		}
		push(@fields, $expr); $pos{$name} = $#fields;
	}
	return unless @fields;
	my @res = $DBI->selectrow_array ("select " . join(',', @fields) . " from $schema.$tableName");
	while (my ($name,$id) = each (%pos)) { $mem->{$name} = $res[$id]; }
	$DBI_DEL->disconnect if $DBI_DEL;
}

sub dbiSpec {
	my %spec = (); if (my $ref = shift) { %spec = %$ref; } else { %spec = &fillDatabaseParams(); }
	my $dbiSpec = sprintf('dbi:Pg:host=%s;port=%i;dbname=%s', @spec{'host','port','catalog'});
	return $dbiSpec;
}


1;

=head1 License

Copyright 2014-2015 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
