/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2014 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.proj.ctx;

import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.matching.external.ProjectMemory.ContextTMXEntry;

import org.silvestrislab.cyclotis.omegat.Cyclotis;

import java.util.*;
import java.sql.*;

public abstract class IntContextMode extends ContextMode<Integer> {
	
	public final void fillEntry (Cyclotis origin, ContextTMXEntry entry, ResultSet data) throws SQLException {
		this.fillEntry (origin, entry, data.getInt("CONTEXT"));
	}
	
	public void setStatementValue (PreparedStatement stmt, int pos, SourceTextEntry key, boolean isDefault) throws SQLException {
		stmt.setInt(pos, toDatabase (key, isDefault));
	}
	
	public static final IntContextMode DEFAULT = new HashContextMode();
	
	public IntContextMode nullWrapper() {
		return new IntContextMode () {
			public Integer defaultValueForSource (String sourceText) { return null; }
			public Integer nonDefaultValueForEntry (SourceTextEntry key) { return IntContextMode.this.nonDefaultValueForEntry(key); }
			public boolean isDefault (Integer value) { return value == null; }
			public boolean isDefault (SourceTextEntry entry, Integer value) { return value == null; }
			public boolean isCorrectKey (SourceTextEntry entry, Integer value) { return IntContextMode.this.isCorrectKey(entry,value); }
			public void setStatementValue (PreparedStatement stmt, int pos, SourceTextEntry entry, boolean isDefault) throws SQLException {
				if (isDefault) stmt.setNull (pos,  java.sql.Types.INTEGER);
				else super.setStatementValue (stmt, pos,entry, false);
			}
			public String toString() { return "null or " + IntContextMode.this.toString(); }
		};
	}
	
	public IntContextMode constWrapper(final int constVal) {
		return new IntContextMode () {
			public Integer defaultValueForSource (String sourceText) { return constVal; }
			public Integer nonDefaultValueForEntry (SourceTextEntry entry) { return IntContextMode.this.nonDefaultValueForEntry(entry); }
			public boolean isDefault (Integer value) { return value == null || value.equals(constVal); }
			public boolean isDefault (SourceTextEntry entry, Integer value) { return value == null || value.equals(constVal); }
			public boolean isCorrectKey (SourceTextEntry entry, Integer value) { return IntContextMode.this.isCorrectKey(entry,value); }
			public void setStatementValue (PreparedStatement stmt, int pos, SourceTextEntry entry, boolean isDefault) throws SQLException {
				if (isDefault) stmt.setInt(pos, constVal);
				else stmt.setInt(pos, this.nonDefaultValueForEntry(entry));
			}
			public String toString() { return "default(" + constVal + ") or " + IntContextMode.this.toString(); }
		};
	}
	
	
	public static IntContextMode forProperties (java.util.Properties propList) {
		ContextMode<?> mode = null;
		try {
			mode = ContextMode.forProperties(propList);
		} catch (Exception e1) {
			mode = DEFAULT;
		}
		try {
			IntContextMode iMode = (IntContextMode) mode;
			String contextDefault = propList.getProperty("context.default");
			if (contextDefault != null) {
				if (contextDefault.equalsIgnoreCase("setNull")) return iMode.nullWrapper();
				else if (contextDefault.toLowerCase().startsWith("const")) 
					try {
						int val = Integer.parseInt (contextDefault.substring(6));
						return iMode.constWrapper(val);
					} catch (Exception valEx) {
						return iMode.constWrapper(0);
					}
			}
			else return iMode;
		} catch (Exception iNo) {
			try {
				final StringContextMode sMode = (StringContextMode) mode;
				return new IntContextMode() {
					public Integer defaultValueForSource (String sourceText) {
						return sMode.defaultValueForSource(sourceText).hashCode();
					}

					public Integer nonDefaultValueForEntry (SourceTextEntry key) {
						return sMode.nonDefaultValueForEntry(key).hashCode();
					}
					
					public boolean isDefault (Integer value) {
						try {
							return 0 == value.intValue();
						} catch (Exception e) {
							return false;
						}
					}
					
					public boolean isDefault (SourceTextEntry key, Integer value) {
						try {
							return sMode.defaultValueForSource (key.getKey().sourceText).hashCode() == value.intValue();
						} catch (Exception e) {
							return false;
						}
					}					
					public boolean isCorrectKey (SourceTextEntry key, Integer value) {
						try {
							return sMode.nonDefaultValueForEntry (key).hashCode() == value.intValue();
						} catch (Exception e) {
							return false;
						}
					}
					
					public String toString () {
						return "hashCode(" + sMode + ")";
					}
					
					public boolean isUnique() {
						return sMode.isUnique();
					}
					
				};
			} catch (Exception iModeEx) {
				return DEFAULT;
			}
		}
		return DEFAULT;
	}
	
}
