#! /usr/bin/perl

=encoding utf-8
=head1 DESCRIPTION

This script implements a deamon connected to directories:

=over 1

=item import-directory

When a TMX file is in this directory it will be imported

=item search-input

When a text file is in this directory, a call to ef-find is done

=item search-output

This directory will receive results from find 

=back

=head1 USAGE

    perl ef-deamon.pl DBI-String <import-dir> <search-input> <search-output>

with DBI-String containing all parameters from L<https://metacpan.org/pod/DBD::Pg>
including user and password if necessary
or only the name of the catalog

=cut

use DBI;
use File::Basename;
use POSIX qw(strftime);

my $db = shift or die "Syntax: $0 DBI-String <import-dir> <search-input> <search-output>";
die "Syntax: $0 DBI-String <import-dir> <search-input> <search-output>" if $db =~ /\.t\w+/ and $db !~ /=/;

my $importDir = shift or die "Syntax: $0 DBI-String <import-dir> <search-input> <search-output>";
my $searchInputDir = shift or die "Syntax: $0 DBI-String <import-dir> <search-input> <search-output>";
my $searchOutputDir = shift or die "Syntax: $0 DBI-String <import-dir> <search-input> <search-output>";

opendir(IMPORT_DIR, $importDir) or die "Directory $importDir not found";
opendir(SEARCH_INPUT_DIR, $searchInputDir) or die "Directory $searchInputDir not found";

our ($importFile, $searchFile);

do {
   until ($importFile or $searchFile) {
       print STDERR "Enter loop\n";
       while ($importFile = readdir(IMPORT_DIR)) {
          next if -d "$importDir/$importFile";
          last if (-f "$importDir/$importFile") and ($importFile =~ /\.tmx$/i);
       }
       rewinddir IMPORT_DIR;
       while ($searchFile = readdir(SEARCH_INPUT_DIR)) {
          next if -d "$searchInputDir/$searchFile";
          last if (-f "$searchInputDir/$searchFile") and ($searchFile =~ /\.te?xt$/i);
       }
       rewinddir SEARCH_INPUT_DIR;
       sleep 5 unless $importFile or $searchFile;
   }
   if ($importFile) { 
       print STDERR strftime('%F %T', localtime(time)), " - Found import file: $importFile\n";
       my %params = (); 
       if (open(CONF, "$importDir/$importFile.conf")) {
           while (<CONF>) { 
               $params{lc(substr($1,0,1))} = $2		# temporary: since we have only collection and doc-name, take first letter only 
                   if /^\s*(\S+)\s*[:=]\s*(.+)\s*$/; 	# accept = or : as separator, and ignore spaces
           }  
           close(CONF); unlink "$importDir/$importFile.conf";
       }
       my $paramStr = $params{c}; $paramStr .= ":$params{d}" if $params{d}; 
       system dirname($0) . "/ef-import.pl '$db' $importDir/$importFile $paramStr";
       print STDERR strftime('%F %T', localtime(time)), " - Imported file: $importFile\n"; unlink "$importDir/$importFile"; $importFile = ''; 
       system dirname($0) . "/ef-stats.pl '$db' > $searchOutputDir/stats";
   }
   if ($searchFile) { 
       print STDERR strftime('%F %T', localtime(time)), " - Found search file: $searchFile\n";
       my $outputFile = $searchFile; $outputFile =~ s/\.te?xt$/.tmx/;
       system dirname($0) . "/ef-find.pl '$db' - -file $searchInputDir/$searchFile > $searchOutputDir/$outputFile";
       print STDERR strftime('%F %T', localtime(time)), " - Executed search file: $searchFile\n"; unlink "$searchInputDir/$searchFile"; $searchFile = ''; 
   }
} while(1);

=head1 LICENSE

Copyright 2013 Silvestris Project (L<http://www.silvestris-lab.org/>)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut


