require 'anguilla'

require 'spongiae/tags'

module Spongiae
   module XLIFF
       
       ##
       # Gets only meta-info about a file, reading beginning
       # use regexes because anguilla parser would read the full file
       def self.read_meta(file,encoding = 'UTF-8')
           res = {}
           File.open(file, "r:#{encoding}") do |f|
               while line = f.gets
                   res['original'] = $1 if line =~ /\soriginal\s*=\s*["'](.+?)["']/
                   res['srcLang'] = $1 if line =~ /\ssource-language\s*=\s*["'](.+?)["']/
                   res['traLang'] = $1 if line =~ /\starget-language\s*=\s*["'](.+?)["']/
                   return res if (res['srcLang'] != nil) and (res['traLang'] != nil)
               end
           end
       end
       
       ##
       # Reads all units, but only the translation. 
       def self.read_translations_map(file)
           callback = Xliff_Callback.new(false,true)
           Anguilla::parse(file,callback)
           return callback.result
       end
       
       ##
       # Reads all units (or only translated), producing translated bilingual objects 
       def self.read_units(file,onlyTra=true,&action)
           if block_given? then
               callback = Xliff_Callback.new(action,onlyTra)
           else
               callback = Xliff_Callback.new(true,onlyTra)
           end
           Anguilla::parse(file,callback)
           return callback.result
       end
       
       
       class Xliff_Callback
           include REXML::StreamListener
           
           def initialize(units,onlyTra)
               @result = {}; @units = units; @onlyTra = onlyTra; @multi_file = false
           end
           
           attr_reader :result
           
           def tag_start(element, attributes)
               if element == 'trans-unit' or element == 'unit' then 
                   @cur_unit = attributes['id']
                   @cur_text = nil
                   @cur_source = ''
                   @cur_tra = nil   # remains nil until there is almost <target>
               elsif element == 'file' then 
                   @multi_file = true if @cur_subfile != nil 
                   @cur_subfile = attributes['original']
               elsif (element == 'target') or (element == 'source') 
                   @cur_text = ''; @cur_tags = []; @stack_tags = []
                   @cur_tra = '' if (element == 'target')   # was nil to make distinction between no target and target empty 
               elsif element == 'g' then
                   tag = Spongiae::Tags::MarkupTag.new; tag.type = 1; tag.id = attributes['id']         
                   @cur_tags <<  Spongiae::Tags::Placeable.new(@cur_text.length, tag)
                   @stack_tags << tag.id
               elsif element == 'x' then
                   tag = Spongiae::Tags::MarkupTag.new; tag.type = 0; tag.id = attributes['id']         
                   @cur_tags <<  Spongiae::Tags::Placeable.new(@cur_text.length, tag)
               elsif element == 'bpt' then
                   tag = Spongiae::Tags::MarkupTag.new; tag.type = +1
                   tag.id = attributes['rid']; tag.id = attributes['id'] if tag.id == nil
                   @cur_tags <<  Spongiae::Tags::Placeable.new(@cur_text.length, tag)
                   @save_cur_text = @cur_text; @cur_text = nil # do not include contents 
               elsif element == 'ept' then
                   tag = Spongiae::Tags::MarkupTag.new; tag.type = -1
                   tag.id = attributes['rid']; tag.id = attributes['id'] if tag.id == nil
                   @cur_tags <<  Spongiae::Tags::Placeable.new(@cur_text.length, tag)
                   @save_cur_text = @cur_text; @cur_text = nil # do not include contents 
               end
           end
           
           def tag_end(element)
               if element == 'source'
                   @cur_source = @cur_source + @cur_text
                   @cur_source = Spongiae::Tags::TaggedString.new(@cur_source,@cur_tags) if @cur_tags != nil and @cur_tags.count > 0
               elsif element == 'target'
                   @cur_tra = @cur_tra + @cur_text
                   @cur_tra = Spongiae::Tags::TaggedString.new(@cur_tra,@cur_tags) if @cur_tags != nil and @cur_tags.count > 0
               elsif element == 'trans-unit' or element == 'unit' then 
                   unless @onlyTra and ((@cur_tra == nil) or (@cur_tra.length == 0))
                       id = @cur_unit; id = "#{@cur_subfile}!!#{id}" if @multi_file
                       if @units.is_a? Proc then 
                            @units.call(Spongiae::Unit::Bilingual.new(@cur_subfile, @cur_unit, {}, @cur_source, @cur_tra))
                       elsif @units then    # true, we wants full units
                           @result[id] = Spongiae::Unit::Bilingual.new @cur_subfile, @cur_unit, {}, @cur_source, @cur_tra
                       else 
                           @result[id] = @cur_tra    # @units = false => only text of the translation
                       end
                   end
                   @cur_source = @cur_tra = nil
               elsif element == 'g' then
                   tag = Spongiae::Tags::MarkupTag.new; tag.type = -1; tag.id = @stack_tags.pop
                   @cur_tags <<  Spongiae::Tags::Placeable.new(@cur_text.length, tag)
               elsif (element == 'bpt') or (element == 'ept') then
                   @cur_text = @save_cur_text; @save_cur_text = nil
               end
           end
           
           def text(text) 
               if @cur_text != nil then @cur_text = @cur_text + text end
           end
       end
       
       class XliffWriter
          def initialize(target, file, options, culter, translations_map)
              @target = target ; @culter = culter; @options = options; @file = file; @translations_map = translations_map
              @target.puts '<?xml version="1.0" encoding="UTF-8"?>'
              @prev_file = file
          end
          
          # Factory to choose between version 1 and 2
          def self.create(target, file, options, culter, translations_map)
              if options['--version'] == nil
                  return Xliff1Writer.new(target, file, options, culter, translations_map)
              elsif options['--version'].to_f < 2.0
                  return Xliff1Writer.new(target, file, options, culter, translations_map)
              else
                  return Xliff2Writer.new(target, file, options, culter, translations_map)                  
              end
          end
       end
       
       class Xliff1Writer < XliffWriter
          def initialize(target, file, options, culter, translations_map)
              super(target, file, options, culter, translations_map)
              @target.puts '<xliff xmlns="urn:oasis:names:tc:xliff:document:1.2">'
              @target.puts "    <file original=\"#{file}\" #{xliff_lang_spec(options)}><body>"    
          end
          
          def xliff_lang_spec(options)
              res = ''
              srcKey = options.keys.select { |item| item =~ /lang/i and item =~ /s(ou)?rc/ }
              res = res + " source-language='#{options[srcKey[0]]}'" unless srcKey.count == 0
              traKey = options.keys.select { |item| item =~ /lang/i and lang =~ /t(ra|arget)/ }
              res = res + " target-language='#{options[traKey[0]]}'" unless traKey.count == 0
              return res 
          end
          
          def write_unit(unit)
              if unit.file != @prev_file and unit.file != nil then
                  @target.puts '    </body></file>'
                  @target.puts "    <file original=\"#{unit.file}\" #{xliff_lang_spec(@options)}><body>"
                  @prev_file = unit.file
              end
              @target.puts "        <trans-unit id=\"#{unit.id}\">"
              @target.puts "            <source>#{unit.srcText.to_xliff(version: 1, segmented: false)}</source>"
              if @culter != nil
                  segments = @culter.cut(unit.srcText)
                  if segments.count > 1 then
                      @target.print  "            <seg-source>"
                      seg_mid = 0
                      segments.each do |txt|
                          seg_mid = seg_mid + 1
                          @target.print "<mrk mtype=\"seg\" mid=\"#{seg_mid}\">#{txt.to_xliff(version: 1, segmented: 'embedded')}</mrk>"                  
                      end
                      @target.puts "</seg-source>"
                  end
              end
              if unit.respond_to? 'traText'
                  tra = unit.traText
                  @target.puts "            <target>#{tra.to_xliff(version: 1, segmented: culter != nil)}</target>" if tra != nil and tra != '' 
              elsif @translations_map[unit.id] != nil then
                  @target.puts "            <target>#{@translations_map[unit.id].to_xliff(segmented: culter != nil)}</target>"
              end
              @target.puts '        </trans-unit>'
          end
          
          def close
              @target.puts '    </body></file>'
              @target.puts '</xliff>'
          end
       end
       
       class Xliff2Writer < XliffWriter
          def initialize(target, file, options, culter, translations_map)
              super(target, file, options, culter, translations_map)
              @target.puts '<xliff xmlns="urn:oasis:names:tc:xliff:document:2.0" version="2.0"  xmlns:fs="urn:oasis:names:tc:xliff:fs:2.0" ' + xliff_lang_spec(options) + '>'
              @target.puts "    <file original=\"#{file}\" }><body>"    
              @prev_file = file
          end
          
          def xliff_lang_spec(options)
              res = ''
              srcKey = options.keys.select { |item| item =~ /lang/i and item =~ /s(ou)?rc/ }
              res = res + " srcLang='#{options[srcKey[0]]}'" unless srcKey.count == 0
              traKey = options.keys.select { |item| item =~ /lang/i and lang =~ /t(ra|arget)/ }
              res = res + " trgLang='#{options[traKey[0]]}'" unless traKey.count == 0
              return res 
          end
          
          def write_unit(unit)
              if unit.file != @prev_file and unit.file != nil then
                  @target.puts '    </file>'
                  @target.puts "    <file original=\"#{unit.file}\">"
                  @prev_file = unit.file
              end
              @target.puts "        <unit id=\"#{unit.id}\">"
              if @culter != nil
                  segments = @culter.cut(unit.srcText)
                  if unit.respond_to? 'traText' then tra = @culter.cut(unit.traText) else tra = nil end
                  i = 0; while i < segments.count
                      @target.puts '            <segment>'
                      @target.puts "                <source>#{segments[i].to_xliff(version: 2, segmented: false)}</source>"
                      @target.puts "                <target>#{tra[i].to_xliff(version: 2, segmented: false)}</target>" if tra != nil and tra.count >= i
                      @target.puts '            </segment>'
                      i = i + 1
                  end
              else  # in XLIFF 2 must be almost one segment
                  @target.puts '            <segment>'
                  @target.puts "                <source>#{unit.srcText.to_xliff(version: 2, segmented: false)}</source>"
                  @target.puts "                <target>#{unit.traText.to_xliff(version: 2, segmented: false)}</target>" if unit.respond_to? 'traText'
                  @target.puts '            </segment>'
              end
              @target.puts '        </unit>'
          end
          
          def close
              @target.puts '    </file>'
              @target.puts '</xliff>'
          end
       end       
       
   end
end
