/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2013-2016 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat;

import org.omegat.util.WikiGet;

import org.silvestrislab.cyclotis.omegat.http.*;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;
import java.util.regex.*;
import javax.swing.*;
import javax.xml.bind.DatatypeConverter;	// needs Java 6 !!!

/** Connection to Cyclotis via HTTP **/
public abstract class HttpCyclotis<T> extends Cyclotis<T> {

	private static final Pattern PATTERN_KEY_VAL = Pattern.compile("^(?:\\;\\s+)?([\\w\\-]+):\\s+(.+)\\r?$");

	// List columns with position in line format's list
	
	public static final int COL_SRC = 0;
	public static final int COL_TRA = 1;

	// -------------------- Data ------------------

	protected String urlFind, urlSave;
	private Map<String, String> headers = null;
	private String dateFormatSpec = null;
	
	public HttpCyclotis (Properties propList) {
		super(propList); // fill name, log
		
		if (propList.getProperty("dateFormat") != null) dateFormatSpec = propList.getProperty("dateFormat");
		for (String propName: propList.stringPropertyNames())
			if (propName.startsWith("header.")) {
				if (headers == null) headers = new HashMap<String,String>();
				headers.put (propName.substring(7), propList.getProperty(propName));
			} 
			else if (propName.equals("http.auth.user"))
				try {
					if (headers == null) headers = new HashMap<String,String>();
					String code = propList.getProperty("http.auth.user") + ":" + propList.getProperty("http.auth.password");
					if (code.toLowerCase().contains("#ask")) {
						JPanel panel = new JPanel();
						panel.add(new JLabel("User:"));
						JTextField userField = new JTextField(code.toLowerCase().startsWith("#ask") ? "" : code.substring(0, code.indexOf(":")), 20);
						panel.add(userField);
						JPasswordField pass = new JPasswordField(20);
						panel.add(new JLabel("Password:"));
						panel.add(pass);
						int option = JOptionPane.showOptionDialog(null, panel, "Credentials for " + this.getMemoryName(),
												 JOptionPane.NO_OPTION, JOptionPane.PLAIN_MESSAGE,
												 null, new String[]{"OK", "Cancel"}, "Cancel");
						if(option == 0) // pressing OK button
							code = userField.getText() + ":" + new String(pass.getPassword());
					}
					headers.put ("Authorization", "Basic " + DatatypeConverter.printBase64Binary(code.getBytes("UTF-8")));
				}
				catch (Exception e) {
				}
			else if (propName.startsWith("http.proxy"))
				System.setProperty (propName, propList.getProperty(propName));

		// Call /info to re-configure plugin. Note: since this is an http call, it must be done after proxy is configured.
		try {
			String url = propList.getProperty("url");
			logMessage ("http", "Parsing Contents from " + url);
			String response = WikiGet.get(url, null, headers);
			url = url.substring (0, url.lastIndexOf("/meta"));
			StringTokenizer tokLines = new StringTokenizer (response, "\n"); String line;
			while (tokLines.hasMoreTokens()) {
				line = tokLines.nextToken(); logMessage ("http-response", "HTTP: " + line);
				Matcher match = PATTERN_KEY_VAL.matcher(line);
				if (match.find()) { 
					if (match.group(1).contains("url") && match.group(2).startsWith("/"))
						propList.setProperty(match.group(1), url + match.group(2));
					else
						propList.setProperty(match.group(1), match.group(2)); 
					logMessage ("config-details", "Read property '" + match.group(1) + "' as '" + match.group(2) + "'");
				}
			}
			logMessage("", "Parsing of " + name + " as " + url + " OK.");
		}
		catch (Exception e) {
			logException(e);
		}
		String defaultFormat = propList.getProperty("default-format"); if (defaultFormat == null) defaultFormat = "line";
		if (propList.getProperty("http.default-format") != null) defaultFormat = propList.getProperty("http.default-format"); // from properties file
		try {
			if (headers == null) headers = new HashMap<String,String>(); headers.put("Accept", (new FormatFactory()).createFormat (defaultFormat).MIME());
		} catch (Exception ex) {
			ex.printStackTrace(); logException(ex);
		}
		this.urlFind = propList.getProperty("url.find"); this.urlSave = propList.getProperty("url.save");
		if (urlFind == null) urlFind = propList.getProperty("find-url"); // coming from http
		if (urlFind == null) urlFind = propList.getProperty("url").replace("/meta", "/tu").replace("info",":cmd");
		urlFind = urlFind.replace(":fmt", defaultFormat).replace("(","").replace(")?","");
		if (urlSave == null) urlSave = propList.getProperty("save-url"); // coming from http
		if (urlSave == null) urlSave = propList.getProperty("url").replace("/meta", "/tu").replace("info",":cmd");		
		urlSave = urlSave.replace(":fmt", defaultFormat).replace("(","").replace(")?","");
		logMessage("config", "URL for find : " + urlFind);
		logMessage("config", "URL for save : " + urlSave);			
	}
	
	public HttpCyclotis (HttpCyclotis ori) {
		super(ori);
		this.headers = ori.headers; this.dateFormatSpec = ori.dateFormatSpec;
		this.urlFind = ori.urlFind; // not save: read-only
	}
	
	/* ---------- http abstraction ----------- */
	
	public abstract T analyzeLine (HttpOutputFormat fmt, Object args);

	protected final List<T> httpGetAll (String command, String lineFormat, String query) throws Exception {
		String completeUrl = urlFind.replace(":cmd",command) + (urlFind.contains("?") ? '&' : '?')
			+ "cols=$" + lineFormat + "&query=" + URLEncoder.encode(query,"UTF-8");
		logMessage("http", "GET " + completeUrl);
		HttpURLConnection conn = (HttpURLConnection) new URL(completeUrl).openConnection();
		try {
			conn.setRequestMethod("GET");
			if (headers != null)
				for (Map.Entry<String, String> en : headers.entrySet()) conn.setRequestProperty(en.getKey(), en.getValue());

            conn.setDoOutput(true);

			if (conn.getResponseCode() != HttpURLConnection.HTTP_OK) throw new WikiGet.ResponseError(conn);
			String contentType = conn.getHeaderField("Content-Type");
			int cp = contentType != null ? contentType.indexOf("charset=") : -1;
			String charset = cp >= 0 ? contentType.substring(cp + 8) : "ISO8859-1";
			logMessage("http", "Response = " + conn.getResponseCode() + " ; content-type = " + contentType + " ; charset = " + charset);
			InputStream in = conn.getInputStream();
			if (dateFormatSpec != null) contentType += ";dateFormat=" + dateFormatSpec + ";";
			HttpOutputFormat fmt = (new FormatFactory()).createFormat (contentType);
			List<T> theList = fmt.readFromGet (this, in, charset);
			logMessage("search", "returned " + theList.size() + " entries");
			return theList;
		} catch (Exception io) {
			logException (io); throw io;
		} finally {
			conn.disconnect();
		}
	}
	
	protected final Iterable<T> httpGetIterable (String command, String lineFormat, String query) throws Exception {
		String completeUrl = urlFind.replace(":cmd",command) + (urlFind.contains("?") ? '&' : '?')
			+ "cols=$" + lineFormat + "&query=" + URLEncoder.encode(query,"UTF-8");
		logMessage("http", "GET " + completeUrl);
		HttpURLConnection conn = (HttpURLConnection) new URL(completeUrl).openConnection();		
		try {
			conn.setRequestMethod("GET");
			if (headers != null)
				for (Map.Entry<String, String> en : headers.entrySet()) conn.setRequestProperty(en.getKey(), en.getValue());

			conn.setDoOutput(true);

			if (conn.getResponseCode() != HttpURLConnection.HTTP_OK) throw new WikiGet.ResponseError(conn);
			String contentType = conn.getHeaderField("Content-Type");
			int cp = contentType != null ? contentType.indexOf("charset=") : -1;
			final String charset = cp >= 0 ? contentType.substring(cp + 8) : "ISO8859-1";
			logMessage("http", "Response = " + conn.getResponseCode() + " ; content-type = " + contentType + " ; charset = " + charset);
			final InputStream in = conn.getInputStream();
			if (dateFormatSpec != null) contentType += ";dateFormat=" + dateFormatSpec + ";";
			final HttpOutputFormat fmt = (new FormatFactory()).createFormat (contentType);			
			return new Iterable<T>() {
				public Iterator<T> iterator() {
					return fmt.iterator (HttpCyclotis.this, in, charset);
				};
			};
		} catch (Exception io) {
			logException (io); throw io;
		} 
	}
	
	
	public DateFormat PSQL_DATE = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSSSS", java.util.Locale.ENGLISH);
	
	/* ------------------ IWritableExternalMemory ---------------*/
	
	public  void registerTranslation(T entryContents) throws Exception {
		try {
			HttpURLConnection connection = (HttpURLConnection) (new URL (this.urlSave)).openConnection();  
			connection.setDoOutput(true);
			connection.setDoInput(true);
			connection.setInstanceFollowRedirects(false); 
			connection.setRequestMethod("POST"); 
			if (headers != null) for (Map.Entry<String, String> en : headers.entrySet()) connection.setRequestProperty(en.getKey(), en.getValue());
			connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded"); 
			connection.setRequestProperty("charset", "utf-8");
			if (headers.get("Accept") == null) connection.setRequestProperty("Accept","text/plain"); // "line" format is text/plain
			
			String args = ""; for (Map.Entry<String,String> e: this.encodeArgs (entryContents).entrySet()) args += e.getKey() + "=" + URLEncoder.encode(e.getValue(),"UTF-8") + "&";
			connection.setRequestProperty("Content-Length", "" + args.getBytes("UTF-8").length);
			connection.setUseCaches (false);
			logMessage("http", "POST " + urlSave);
			logMessage("http", args);
			
			OutputStreamWriter wr = new OutputStreamWriter(connection.getOutputStream(), "UTF-8");
			wr.write(args); wr.flush(); wr.close();		
			
			BufferedReader rd = new BufferedReader(new InputStreamReader(connection.getInputStream()));
			String line; int index; while ((line = rd.readLine()) != null) {
				if (line.startsWith(";")) 
					if ((index = line.indexOf("ERROR")) > 0) {
						logMessage ("", line.substring (index + 6)); // log, but continue
						throw new Exception (getProviderName() + " / " + getMemoryName() +  ": " + line.substring (index + 6));
					}
					else if ((index = line.indexOf("FAIL")) > 0) {
						logMessage ("", line.substring (index + 5)); // log, but continue					
						throw new Exception (getProviderName() + " / " + getMemoryName() +  ": " + line.substring (index + 5));
					}
			}
			
			connection.disconnect();
			logEntry("update-contents", entryContents, false);
		} catch (IOException e) {
			logException (e); throw e;
		}
	}
	
	protected abstract Map<String,String> encodeArgs (T entry);
			
}
