/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2014-2015 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/
package org.silvestrislab.cyclotis.omegat.proj;

import org.omegat.core.matching.external.ProjectMemory.ContextTMXEntry;
import org.omegat.core.matching.external.ProjectMemory;
import org.omegat.core.events.IProjectEventListener;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.data.TMXEntry;
import org.omegat.core.data.EntryKey;
import org.omegat.core.CoreEvents;
import org.omegat.util.TMXProp;

import java.util.WeakHashMap;
import java.util.Properties;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.List;


/** This class describes whenever we want to store entry key or number in the properties **/
public class EntryKeyStore {
	private boolean store_linked = true, store_num = true, store_key = false, store_whenDefault = false;

	public EntryKeyStore(Properties propList) {
		if ("false".equals(propList.getProperty("entryKey.store.linked"))) store_linked = false;
		if ("false".equals(propList.getProperty("entryKey.store.entryNum"))) store_num = false;
		if ("true".equals(propList.getProperty("entryKey.store.entryKey"))) store_key = true;
		if ("true".equals(propList.getProperty("entryKey.store.whenDefault"))) store_whenDefault = true;
		CoreEvents.registerProjectChangeListener(new IProjectEventListener() {
			public void onProjectChanged(PROJECT_CHANGE_TYPE eventType) { 
				switch (eventType) {
					case CLOSE: case CREATE: case LOAD:
						cache.clear(); 
				}
			}
		});
	}
	
	public String toString() {
		return (store_linked ? "link;" : "no-link;")
			+ (store_num ? "entry-number": "no-entry-number;")
			+ (store_key ? "entry-key": "no-entry-key")
			+ (store_whenDefault? " (non-default only)" : "(including default)");
	}

	/** Build a list of properties : can contain linked status, entry number, and/or key **/
	public List<TMXProp> buildProperties (ContextTMXEntry entry) {
		List<TMXProp> props = new ArrayList<TMXProp>();
		if (store_linked && (entry.linked != null)) props.add(new TMXProp("externalLinked", entry.linked.toString()));
		if (store_num) props.add(new TMXProp("entryNum", Integer.toString(entry.entryNum)));
		if (store_key) {					
			EntryKey key = org.omegat.core.Core.getProject().getAllEntries().get(entry.entryNum - 1).getKey();
			props.add(new TMXProp("id", key.id));
			props.add(new TMXProp("prev", key.prev));
			props.add(new TMXProp("next", key.next));
			props.add(new TMXProp("path", key.path));
			props.add(new TMXProp("file", key.file));
		}
		return props;
	}
	
	/**
	 * Whenever we should write the properties:
	 * in the properties file is decided whenever properties are stored for default entries or not
	 **/
	public boolean shouldWrite(ContextTMXEntry entry) {
		return (! (store_whenDefault ^ entry.isDefault));
	}
	
	private WeakHashMap<EntryKey, Integer> cache = new WeakHashMap<EntryKey,Integer>();
	
	/**
	 * Sets context properties from current keyStore, if possible
	 **/
	public void fillEntry(ContextTMXEntry entry) {
		try { entry.linked = TMXEntry.ExternalLinked.valueOf (entry.getPropValue("externalLinked")); } catch (Exception e) { }
		HashMap<String,String> properties = new HashMap<String,String>();
		for (TMXProp prop: entry.otherProperties) properties.put(prop.getType(), prop.getValue());
		EntryKey eKey = new EntryKey (
			properties.get("file"), entry.source,
			properties.get("id"), 
			properties.get("prev"), properties.get("next"),
			properties.get("path")		
		);
		entry.entryNum = -1; List<SourceTextEntry> projectEntries = org.omegat.core.Core.getProject().getAllEntries();
		// entry key values have priority over entry number, because it can differ if you do not have the same source files in the project
		if (eKey.hashCode() != entry.source.hashCode()) { // entry really contains something
			if (cache.get(eKey) != null)
				if (projectEntries.get (cache.get(eKey) - 1).getKey().equals(eKey)) {
					entry.entryNum = cache.get(eKey); return;
				}
				else cache.remove(eKey); // de-synchronisation
			for (SourceTextEntry sourceEntry: projectEntries)
				if (sourceEntry.getKey().equals(eKey)) { cache.put(eKey, entry.entryNum  = sourceEntry.entryNum()); return; }
		}
		// if entry key is not stored, then we try to use entry number but only if it is valid
		if (entry.entryNum < 0)
			try { 
				entry.entryNum = Integer.parseInt(entry.getPropValue("entryNum")); 
				if (! projectEntries.get(entry.entryNum - 1).getKey().sourceText.equals(entry.source)) entry.entryNum = -1; // unsynchronized project
				cache.put(eKey, entry.entryNum);
			}
			catch (Exception e) { entry.entryNum = -1; }
	}

}