/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2014 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.proj.ctx;

import org.omegat.core.Core;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.matching.external.ProjectMemory.ContextTMXEntry;

import org.silvestrislab.cyclotis.omegat.Cyclotis;

import java.util.*;
import java.sql.*;

public abstract class ContextMode<T> {
			
	public abstract T defaultValueForSource (String sourceText);
	public abstract T nonDefaultValueForEntry (SourceTextEntry key);
	public abstract boolean isDefault (T value);
	public abstract boolean isDefault (SourceTextEntry key, T value);
	public abstract boolean isCorrectKey (SourceTextEntry key, T value);
	public abstract void setStatementValue (PreparedStatement stmt, int pos, SourceTextEntry key, boolean isDefault) throws SQLException;	
	
	public boolean isUnique() { return false; } // may be overridden
	
	public final T toDatabase (int entryNum, boolean isDefault) {
		SourceTextEntry entry = Core.getProject().getAllEntries().get (entryNum - 1);
		if (isDefault) return defaultValueForSource(entry.getKey().sourceText);
		else return nonDefaultValueForEntry(entry);
	}	

	public final T toDatabase (SourceTextEntry entry, boolean isDefault) {
		if (isDefault) return defaultValueForSource(entry.getKey().sourceText);
		else return nonDefaultValueForEntry(entry);
	}	
	
	public void fillEntry (Cyclotis origin, ContextTMXEntry entry, ResultSet data) throws SQLException {
		this.fillEntry (origin, entry, (T) data.getObject("CONTEXT")); // To be overridden
	}
	
	public void fillEntry (Cyclotis origin, ContextTMXEntry entry, T contextValue) {
		List<SourceTextEntry> entries = Core.getProject().getAllEntries();
		if (entry.entryNum > 0) {	// already found thanks to entry key store
			entry.isDefault = (this.isDefault (entries.get(entry.entryNum - 1), contextValue));
			origin.logMessage("context", "Entry #" + entry.entryNum + " with context " + contextValue + " is " + (entry.isDefault ? "default" : "non-default"));
		} else if (this.isDefault (contextValue)) {
			origin.logMessage("context", "" + contextValue + " is for default, searching only entry number");		
			entry.entryNum = -1; //may have been set by try
			for (SourceTextEntry se: entries)
				if (entry.source.equals(se.getSrcText()) && Core.getProject().getTranslationInfo(se).defaultTranslation) { 
					entry.entryNum = se.entryNum(); entry.isDefault = true; break; 
				}
		} else {	// not found in entry key store: either because there is no key store, or because it is an orphaned entry
			origin.logMessage("context", "Searching entry for context " + contextValue);		
			// Find the source entry
			entry.entryNum = -1; //may have been set by try
			for (SourceTextEntry se: entries)
				if (entry.source.equals(se.getSrcText())) // avoid same context used for another entry, which could occur if database is also used as memory
					if (isCorrectKey (se, contextValue)) { 
						entry.isDefault = isDefault(se, contextValue); 
						if (entry.isDefault && Core.getProject().getTranslationInfo(se).defaultTranslation) { entry.entryNum = se.entryNum(); break; }
					}
					else if (isDefault(se, contextValue)) { 
						entry.isDefault = true; 
						if (Core.getProject().getTranslationInfo(se).defaultTranslation) entry.entryNum = se.entryNum(); 
					} // do not break, because it can be contraditory later
			origin.logMessage("context", "From scan : " + entry.entryNum);
			if (entry.entryNum == -1) { 	// scan failed: orphan entry
				entry.isDefault = (contextValue.equals (defaultValueForSource(entry.source)));
				origin.logMessage("context-detail", "Default value for '" + entry.source +"' is " + defaultValueForSource(entry.source) + "; match = " + entry.isDefault);
			}
		}		
	}

	public static ContextMode<?> forProperties (java.util.Properties propList) throws Exception {
		String name = propList.getProperty("context.store").toUpperCase().replace(' ','_');
		if (name.equals ("ENTRY_NUM")) return new EntryNumContextMode(); // please use it only if OmegaT says "isSupportDefaultTranslations = false"
		if (name.equals ("HASH_CODE")) return new HashContextMode();
		if (name.equals ("FULL_STRING")) return new FullStringContextMode();
		if (name.equals ("DIGEST")) {
			String algo = propList.getProperty("context.store.algo").toUpperCase();
			if (algo == null) algo = "MD5";
			return new DigestStringContextMode (algo);
		}
		return null;
	}
	
}
