/**************************************************************************
 Silvestris Cyclotis - Intermediate real-time translation memory server
 
 Copyright (C) 2013-2016 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.tm;

import org.silvestrislab.cyclotis.omegat.PostgresqlCyclotisTMX;

import org.omegat.core.matching.external.IWritableExternalMemory;
import org.omegat.core.matching.external.IBrowsableMemory;
import org.omegat.core.search.SearchExpression;
import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.util.Language;
import org.omegat.util.Preferences;
import org.omegat.util.TMXProp;

import java.util.*;
import java.sql.*;

public class PostgresqlMemory extends PostgresqlCyclotisTMX<PrepareTMXEntry> implements IWritableExternalMemory, IBrowsableMemory {

	private SaveConditions conditions;
	private String exclude = "";
	
	public PostgresqlMemory (Properties propList) throws SQLException, ClassNotFoundException {
		super(propList); // fill name, log
		
		if (propList.getProperty("update") != null) {
			String updateString = propList.getProperty("update");
			if (! (updateString.equalsIgnoreCase("false"))) {
				this.conditions = SaveConditions.forProperty(updateString);
				String table = propList.getProperty ("table"); if (table == null) table = "MEM";		
				String author = propList.getProperty ("author");
				if (author == null) author = Preferences.getPreferenceDefault(Preferences.TEAM_AUTHOR, System.getProperty("user.name"));				
				pInsert = this.getInsertStatement();
				pInsert.setString (3, author);
			}
			else conditions = null;
		}
		else if (propList.getProperty("table.exclude") != null) {
			this.tableWrite = propList.getProperty("table.exclude");
			this.conditions = null;
			
			if (this.mem_id instanceof Integer) { ResultSet set = getInfoResultSet("MEM_ID"); if (set.next()) exclude = " and MEM_ID != " + set.getInt(1); set.close(); } 
			if (this.mem_id instanceof Long) { ResultSet set = getInfoResultSet("MEM_CODE"); if (set.next()) exclude = " and MEM_CODE != " + set.getLong(1); set.close(); } 
			if (this.mem_id instanceof String) { ResultSet set = getInfoResultSet("MEM_PATH"); if (set.next()) exclude = " and MEM_PATH != '" + set.getString(1) + "'"; set.close(); } 
		}
		else conditions = null;
		System.out.println (getMemoryName() + ((this.conditions == null) ? " update = false" :  " update condition = " + conditions));
	}
	
	public PostgresqlMemory (PostgresqlCyclotisTMX ori) {
		super(ori);
		// do not copy save conditions, because we are read-only
	}	
	
	/* ------------------ SQL -------------------- */
	
	@Override protected String selectCondition() { return "src % ?" + exclude; }
	
	/* ------------------ IExternalMemory ---------------*/	
	
	protected final PreparedStatement selectStatement (String table) throws SQLException {
		logMessage("sql", "Select statement = select * from " + table + " where src % ?");
		return connection.prepareStatement ("select * from " + table + " where src % ?");
	}
	
	public List<PrepareTMXEntry> findMatchingTranslations (Language sLang, Language tLang, String text, int minScore, int maxCount) throws Exception {
		logMessage("search", "Searching for <" + digest(text) + "> '" + text + "'");
		getSelectStatement().setString (1, reduceWords(text));
		return retreiveQuery(getSelectStatement());
	}

	@Override
	protected PrepareTMXEntry buildEntry (ResultSet set) throws SQLException {
		PrepareTMXEntry entry = new PrepareTMXEntry ();
		entry.source =  reformatText(set.getString("SRC"),true);
		entry.translation = reformatText(set.getString("TRA"),true);
		try {
			// Retreive fields in order : after one is null, the following ones will be null as well
			entry.creator = set.getString("AUTHOR");
			entry.creationDate = set.getTimestamp("DATE").getTime();
			entry.changer = set.getString("CHANGER");
			entry.changeDate = set.getTimestamp("CHANGEDATE").getTime();
			entry.note = set.getString("NOTE");
		} catch (Exception e) {
			// OK, we do with the fields which did not launch exception
		}
		if (this.col_props != null) {
			List<TMXProp> read = this.col_props.read(set.getObject("PROPS"));
			try { read.add (new TMXProp ("db.table.name", set.getString("MEM_NAME"))); } catch (Exception e1) { read.add (new TMXProp ("db.table.name", tableRead)); }
			entry.otherProperties = store.fromDatabase(read, entry);
		}
		else entry.otherProperties = store.fromDatabase(null, entry); // some properties may not depend from db properties (but from ${...} variables)
		logEntry("search-results", entry, true);
		return entry;
	}

	/* ------------------ IWritableExternalMemory ---------------*/
	
	public boolean isWriteMode() {
		return (conditions != null);
	}
	
	protected List<String> insertFields() {
		List<String> res = new ArrayList<String>(4);
		res.add("SRC"); res.add("TRA"); res.add("AUTHOR");
		if (this.col_props != null) res.add("PROPS");
		if (this.hasNote) res.add("NOTE");
		if (this.contextMode != null) res.add("CONTEXT");
		if (this.mem_id != null) 
			if (this.mem_id instanceof Integer) res.add("MEM_ID");
			else if (this.mem_id instanceof Long) res.add("MEM_CODE");
			else if (this.mem_id instanceof String) res.add("MEM_PATH");
		return res;
	}
	
	public boolean mustWrite(PrepareTMXEntry entryContents, SourceTextEntry entrySource) {
		return conditions.mustWrite (this, entryContents, entrySource);
	}		
	
	public  void registerTranslation(Language sLang, Language tLang, PrepareTMXEntry entryContents, SourceTextEntry entrySource) throws Exception {
		if (this.col_props != null) 
			getInsertStatement().setObject (4, this.col_props.write (store.toDatabase(entryContents, entrySource)));
		if (this.hasNote) {
			int idx = (this.col_props == null) ? 4 : 5;
			if (entryContents.note == null) getInsertStatement().setNull (idx, java.sql.Types.VARCHAR); else getInsertStatement().setString(idx, entryContents.note);
		}
		if (this.contextMode != null) {
			int idx = 4; if (this.col_props != null) idx++; if (this.hasNote) idx++;
			contextMode.setStatementValue (getInsertStatement(), idx, entrySource, false);			
		}
		registerTranslation(entryContents);
	}
	
	/* ------------------ ISearchableMemory ---------------*/	
	
	private boolean hasDate = false, hasChangeDate = false;
	
	protected void registerColumn(ResultSet set, java.util.Properties propList) throws SQLException {
		super.registerColumn(set, propList);
		if (set.getString("column_name").equalsIgnoreCase("date")) hasDate = true;
		if (set.getString("column_name").equalsIgnoreCase("changedate")) hasChangeDate = true;
	}	
	
	public List<PrepareTMXEntry> search(SearchExpression expression) throws Exception {
		logMessage("search", "Searching for '" + expression.text + "'");
		StringBuffer sql = new StringBuffer(getSelectFrom() + " where (0=1");
		List<Object> values = new ArrayList<Object>();
		if (expression.searchSource) sql.append(" or ").append(searchExpressionText(expression, "src", expression.text, values));
		if (expression.searchTarget) sql.append(" or ").append(searchExpressionText(expression, "tra", expression.text, values));
		if (expression.searchNotes) sql.append(" or ").append(searchExpressionText(expression, "note", expression.text, values));
		sql.append(")");
		if (expression.searchAuthor) { sql.append( " and author = ? "); values.add(expression.author); }
		if (expression.searchDateAfter) {
			if (hasChangeDate) sql.append(" and (changedate <= ? or (changedate is null and date <= ?))"); else sql.append("and date <= ?");
			values.add (new java.sql.Date(expression.dateAfter)); values.add (new java.sql.Date(expression.dateAfter));
		}
		if (expression.searchDateBefore) {
			if (hasChangeDate) sql.append(" and (changedate >= ? or (changedate is null and date >= ?))"); else sql.append("and date >= ?");
			values.add (new java.sql.Date(expression.dateBefore)); values.add (new java.sql.Date(expression.dateBefore));
		}
		return executeSearch (sql.toString(), values);
	}
	
	private static String searchExpressionText (SearchExpression expression, String var, String txtVal, List<Object> values) {
		switch (expression.searchExpressionType) {
			case EXACT: 
				values.add (txtVal); // only once
				if (expression.caseSensitive) return "position(lower(?) in lower(" + var + ")) > 0"; else return "position(? in " + var + ") > 0";
			case KEYWORD: 
			{
				StringBuffer buf = new StringBuffer();
				for (String pattern: txtVal.split("\\s")) buf.append(" or ").append(searchExpressionText(expression, var, pattern, values));
				return buf.toString().substring(4);
			}
			case REGEXP:
				values.add (txtVal); // only once
				return var + (expression.caseSensitive ? " ~ " : " ~* ") + "?";
		}
		return null;
	}
		
	private List<PrepareTMXEntry> executeSearch(String sql, List<Object> values) throws SQLException {
		logMessage("sql", "Select statement for search: " + sql);
		PreparedStatement searchStmt = connection.prepareStatement (sql);
		try {
			for (int i = 0; i < values.size(); i++)
				if (values.get(i) instanceof String) searchStmt.setString(i + 1, values.get(i).toString());
				else if (values.get(i) instanceof java.sql.Date) searchStmt.setDate(i + 1, (java.sql.Date) values.get(i));
			return retreiveQuery(searchStmt);
		} finally {
			try { searchStmt.close(); } catch (Exception e) {}
		}
	}
	
	/* ------------------ ISearchableMemory ---------------*/	
	
	public Iterable<PrepareTMXEntry> getEntries() throws SQLException {
		logMessage("search", "Searching for all entries");
		PreparedStatement st = connection.prepareStatement(getSelectFrom());
		return iterateQuery(st); // NOTE: do not close st, because iterator may continue to use the result set
	}
	
}