require 'date'

module Spongiae
   module Unit
      
       # ----------------- For unilingual files ------------
       
       class Unilingual
           attr_reader :props, :text, :id, :file
           
           def initialize(file, id, props, text)
               @props = props; @id = id; @text = text; @file = file
           end
           
           def srcText() @text end
       end
       
       # ----------------- For TMX -------------------
       
       ##
       # An array of key-value pairs; 
       # looks like an hash but can contain multiple times same value
       class PropertiesArray
           def initialize(from = nil) 
               @keys = []; @vals = []
               if from != nil then
                   from.each { |k,v| @keys << k; @vals << v  }   # works for Hash or PropertiesArray
               end               
           end
          
           # Do an each |k,v| loop
           def each
               i = 0
               while i < @keys.count  
                   yield @keys[i],@vals[i] 
                   i = i + 1
               end
           end
           
           def each_with_index
               i = 0
               while i < @keys.count  
                   yield @keys[i],@vals[i],i
                   i = i + 1
               end
           end           
           
           def []=(k,v) 
               if k.is_a? Integer then  # index
                   @vals[k] = v
               else # real key - string or symbol
                   @keys << k; @vals << v; 
               end
           end
               
           def [](k)
               if k.is_a? Integer then   # index
                   return @vals[k] 
               else # real key - string or symbol
                   i = 0
                   while i < @keys.count  
                       return @vals[i] if @keys[i] == k
                       i = i + 1
                   end
               end
           end
           
           # If integer, delete key/value pair from position
           # If string, delete all occurences of a key
           def delete!(key)
               if key.is_a? Integer then    # index'th value/pair
                   @vals.delete_at(key); @keys.delete_at(key)
               else     # delete all occurences of a key
                   i = 0
                   while i < @keys.count  
                       delete!(i) while key == @keys[i]
                       i = i + 1
                   end
               end
           end 
           
           # If val is a sub, apply for each value
           # else replace first value
           # return number of replacements done
           def replace!(key,expr)
               expr1 = expr; expr1 = Proc.new { |item| expr } unless expr.is_a? Proc
               count = 0
               for i in 0 .. @keys.count do
                   if @keys[i] == key then
                        @vals[i] = expr1.call(@vals[i])
                        count = count + 1
                   end
                end
                return count
           end
           
           def copyWith(key,val)
               res = PropertiesArray.new(self)
               res[key] = val unless res.replace!(key,val) > 0
               return res
           end
           
           def to_s
               res = "{"
               for i in 0 .. @keys.count do
                    res += @keys[i].to_s
                    res += " = "
                    res += @vals[i].to_s
                    res += ", "
               end
               res += "}"; return res
           end
       end
       
       class Variant
           attr_reader :props, :text, :lang, :attr
           
           ##
           # Initialize a tuv
           # Parameters: 
           #    attr : contains either a language as string, or hash of <tuv> attributes
           #    props : contains either hash of properties, or list of key-value pairs
           def initialize(attr, props, text)
               @props = PropertiesArray.new(props); @text = text; @attr = attr
               if attr.is_a? String then 
                   @lang = attr
                   @attr = { 'xml:lang' => @lang }
               else
                   @lang = attr['lang']
                   @lang = attr['xml:lang'] if lang == nil
                   @attr = attr.select { |key, value| (key =~ /^(creat|change)/) and (value != nil) }
                   @attr['creationdate'] = Date.parse(@attr['creationdate']).strftime('%Y%m%dT%H%M%SZ') if @attr.has_key? 'creationdate'
                   @attr['changedate'] = Date.parse(@attr['changedate']).strftime('%Y%m%dT%H%M%SZ') if @attr.has_key? 'changedate'
               end
           end
           
           def creator() @attr['creationid'] end
           def creationdate(fmt = nil) fmt == nil ? @attr['creationdate'] : Date.parse(@attr['creationdate']).strftime(fmt) end
           def changer() @attr['changeid'] end
           def changedate(fmt = nil) fmt == nil ? @attr['changedate'] : Date.parse(@attr['changedate']).strftime(fmt) end
       end
       
       class Multilingual
           attr_reader :props, :variants, :id, :attr
           
           ##
           # Initialize a translation unit
           # Parameters: 
           #    attr : contains either a tuid (string, number, ..), or hash of <tu> attributes
           #    props : contains either hash of properties, or list of key-value pairs
           def initialize(attr, props)
               @props = PropertiesArray.new(props); @attr = attr
               if attr.is_a? Hash then 
                   @id = attr['tuid']
                   @attr = attr.select { |key, value| (key =~ /^(creat|change|xml:lang|lang)/) and (value != nil) }
                   @attr['creationdate'] = Date.parse(@attr['creationdate']).strftime('%Y%m%dT%H%M%SZ') if @attr.has_key? 'creationdate'
                   @attr['changedate'] = Date.parse(@attr['changedate']).strftime('%Y%m%dT%H%M%SZ') if @attr.has_key? 'changedate'
               else # string or number
                   @id = attr
                   @attr = { 'tuid' => @id }
               end               
               @variants = { } # hash of Variants per language
           end
           
           def creator() @attr['creationid'] end
           def creationdate(fmt = nil) fmt == nil ? @attr['creationdate'] : Date.parse(@attr['creationdate']).strftime(fmt) end
           def changer() @attr['changeid'] end
           def changedate(fmt = nil) fmt == nil ? @attr['changedate'] : Date.parse(@attr['changedate']).strftime(fmt) end

           def split(srcLang)
              res = Hash.new unless block_given?
              variants.each do |lg,tu|
                  if lg != srcLang then
                      part = Spongiae::Unit::Multilingual.new(@id,@props)
                      part.variants[srcLang] = @variants[srcLang]
                      part.variants[lg] = tu
                      if block_given? then yield lg,part else res[lg] = part end
                  end
              end
              return res unless block_given?
           end
           
           # For files coming from OmegaT: rebuild id based on props
           def rebuild_id_omegat!()
               digest = Digest::SHA1.new
               digest << @variants[@props[:srcLang]].text
               ['id','path','prev','next'].each { |item| digest << @props[item] if defined? @props[item] and @props[item] != nil }
               @id = digest.hexdigest
           end
       end

       # ----------------- For strictly bilingual formats like XLIFF or PO -------------------
       
       class Bilingual
           attr_reader :props, :srcText, :traText, :id, :file
           
           def initialize(file, id, props, srcText, traText)
               @props = props; @id = id; @srcText = srcText; @traText = traText; @file = file
           end
       end
   end
end
