=head1 DESCRIPTION

Objects blessed using this class contain information about a database table.

=head1 METHODS

=cut
package Silvestris::Cyclotis::Database::Table;

use Dancer ':syntax';
use Dancer::Plugin::Database;

our %CACHE;

=head2 $ref = Silvestris::Cyclotis::Database::Table->find ($db, $table,$schema)

Retreives info from the database and builds the object.

=cut
sub find {
    shift; my ($db, $table,$schema) = @_; $db ||= '';
    $table = $1 if $table =~ m!<((\w+\.)?\w+)$!;
	my $distDbName = config->{plugins}{Database}{connections}{$db}{database} || config->{plugins}{Database}{database};
	my $sql = "select table_schema, table_name, column_name, udt_schema, udt_name, is_nullable, data_type from INFORMATION_SCHEMA.COLUMNS where table_catalog = ?";
	if ($table = lc($table)) {	# We want only this table (whose name must be case insensitive), as a hash reference
		($schema, $table) = ($1, $2) if $table && ($table =~ /^(\w+)\.(\w+)$/); $schema ||= 'public';
		if ($CACHE{"$db.$schema.$table"}) { $CACHE{"$db.$schema.$table"}{'last-internal-use'} = time(); return $CACHE{"$db.$schema.$table"}; }
		$sql .= " and table_schema=? and table_name=?"; my $st = database($db)->prepare_cached($sql) or die $DBI::errstr;
		$st->execute ($distDbName, lc($schema),  lc($table)) or die $DBI::errstr;
		print STDERR "$sql ($db -> $distDbName, $schema,$table)\n" if config->{debug}{sql};
		Silvestris::Cyclotis::Database::Table->parse_meta ($st, $db);
		$CACHE{"$db.$schema.$table"}{'last-internal-use'} = time() if defined $CACHE{"$db.$schema.$table"};
		if (wantarray) { return ( "$schema.$table" => $CACHE{"$db.$schema.$table"} ); } else { return $CACHE{"$db.$schema.$table"}; }
	} else {	# We want more than one table, as name=>hashref hash table
		$sql .= " and table_schema=?" if $schema; my $st = database($db)->prepare_cached($sql) or die $DBI::errstr;
		if ($schema) { $st->execute($distDbName, lc($schema)); } else { $st->execute ($distDbName); }
		print STDERR sprintf("$sql (%s,%s)\n", (config->{plugins}{Database}{database}, lc($schema))) if config->{debug}{sql};
		return Silvestris::Cyclotis::Database::Table->parse_meta ($st, $db);		
	}
}

sub parse_meta {
	my (undef, $st, $db) = @_; my %RES = (); 
	# Step 1. Get fields from std view INFORMATION_SCHEMA.COLUMNS
	while (my ($table_schema,$table_name,$column,$type_schema,$type_name, $nullable, $dataType) = $st->fetchrow_array) { 
		if (my $temp = $CACHE{"$db.$table_schema.$table_name"}) {  $RES{"$table_schema.$table_name"} = $temp; $temp->{'last-internal-use'} = time(); next; }
		else { 
			my $Table = $RES{"$table_schema.$table_name"} ||= bless { db => $db }; 
			$Table->{fields}{$column} = "$type_schema.$type_name"; $Table->{fields}{$column} .= '?' if $nullable =~ /YES/i;  $Table->{'last-internal-use'} = time(); 
			$Table->{fields}{$column} = Silvestris::Cyclotis::Database::CompositeType->find ($db, $type_schema,$type_name) if ($dataType =~ /USER-DEFINED/) and ($type_name !~ /^hstore|json$/i);
		}
	}
	# Step 2. Get meta_info for each table
	my $metaInfoView = config->{'meta-info'}{view} || 'public.meta_info'; $metaInfoView = "public.$metaInfoView" unless $metaInfoView =~ /\./;
	my ($storeType); my $idField = undef;
	if ($CACHE{"$db.$metaInfoView"}) { 
		$storeType = $CACHE{"$db.$metaInfoView"}{fields}{propsstore}; 
		($idField) = grep { /^mem_(\w+)$/ } keys %{$CACHE{"$db.$metaInfoView"}{fields}};
	} else {
		my ($minfoSchema,$mInfoView) = split(/\./, $metaInfoView);
		my $distDbName = config->{plugins}{Database}{connections}{$db}{database} || config->{plugins}{Database}{database};
		($storeType) = database($db)->selectrow_array('select udt_name from INFORMATION_SCHEMA.COLUMNS '
                  . " where table_catalog = '$distDbName'"
                  . "   and table_schema = '$minfoSchema' and table_name='$mInfoView' and column_name = 'propsstore'");
		($idField) = database($db)->selectrow_array('select column_name from INFORMATION_SCHEMA.COLUMNS '
                  . " where table_catalog = '$distDbName'"
                  . "   and table_schema = '$minfoSchema' and table_name='$mInfoView' and column_name like 'mem_%'");				  
	}
	$st = database($db)->prepare_cached("select * from $metaInfoView where table_name = ? and table_schema = ?");
	while (my ($tabname, $tabcontents) = each(%RES)) { 
		$CACHE{"$db.$tabname"} = $tabcontents; # now we are sure it is complete	
		my ($schema,$table) = split(/\./, $tabname);
		if ($st->execute (lc($table), lc($schema))) {
			while (my $ref = $st->fetchrow_hashref) { 
				while (my ($k,$v) = each(%$ref)) { 
					if ($k =~ /propsStore/i) {
						$tabcontents->{$k} = Silvestris::Cyclotis::Database::Table->expand_col($v, $storeType) if $storeType;	
					} else {
						$tabcontents->{$k} = $v; 
					}
				} 
			}
		} else {
			print "ERROR : $DBI::errstr\n";
		}
	}
	return %RES;
}

=head2 $table = $self->standard_parent()

Tries to give the standard parent from the table, either stored in the memory, or by calling function in the database

=cut
sub standard_parent {
	my $table = shift; my $res = lc($table->{table_schema} . '.' . $table->{table_name});
	return $table->{std_parent} if $table->{std_parent};
	# Try to find parent from SQL
	my $st = database($table->{db})->prepare("SELECT parent_table(?)");
	until ($res =~ /^public\.(mem|glos)/i) {
		$st->execute ($res);
		($res) = $st->fetchrow_array();
		return '' unless $res =~ /^\w+\.\w+$/;
	}
	# Check that parent has same fields as child
	if (my $parent = find Silvestris::Cyclotis::Database::Table($table->{db}, $res)) {
		my $localFields = $table->{fields}; my $parentFields = $parent->{fields};
		return '' if scalar keys (%$localFields) != scalar keys (%$parentFields);
		foreach my $fieldName (keys (%$localFields)) { return '' if not $parentFields->{$fieldName}; }
	}
	# If all is OK we can return
	return $table->{std_parent} = $res;
}


=head2 $boolean = $self->has_structured_content

True if almost one line should be expanded. In Cyclotis this means that we have context or properties

=cut
sub has_structured_content {
	return $_[0]->{fields}{context} || $_[0]->{fields}{props};
}

=head2 $self->expand($line)

Convert structured fields from database string to perl Hash

Fields can be stored in the database as text (encoded), hstore or json. 
But in any case, Perl receives a string. Here we convert them back to a hash.

=cut
sub expand {
	my ($self, $line) = @_;
	while (my ($key, $type) = each (%{$self->{fields}})) { 
		next if ref($line->{$key});
		$line->{$key} = $self->expand_col ($line->{$key}, $type) if $line->{$key};
	}
}

sub expand_col {
	my (undef, $val, $type) = @_;
	if ($type =~ /hstore/) {
		my $txt = "{ $val }"; $txt =~ s/\$/\\\$/g; my $hash; eval "\$hash = $txt;";
		$val = $hash if $hash;
		$val = $hash->{'!!value'} if $hash->{'!!value'};
	} elsif ($type =~ /json/) {
		my $txt = $val; $txt =~ s/\$/\\\$/g; 
		$txt =~ s/"(.+?)"\s*:\s*"(.+?)"/'$1' => '$2'/gs;    # JSON to Perl/Dump; string to string
		$txt =~ s/"(.+?)"\s*:\s*([\+\-]?\d+(?:\.\d+)?(?:[eE][\+\-]\d+)?)/'$1' => $2/gs;    # JSON to Perl/Dump; string to number	
		my $hash; eval "\$hash = $txt;";
		$val = $hash if $hash;
	} elsif (ref($type) =~ /Composite/) {	# composite type
		if ($val =~ /^\((.+)\)$/) {
			my @tab = split(',', $1); my %h = ();
			foreach my $name (@{$type->{fields}}) { $h{$name} = shift(@tab); }
			$val = \%h;
		}
	} elsif ($val and $val =~ /^TABLE:/) {
		$val = substr($val, 6); my %res;
		foreach my $item (split(/\&/,$val)) { $res{$1} = $2 if $item =~ /^(.+)=(.+)$/; }
		$val = \%res;
	}
	return $val;
}

=item $self->unexpand($line)

Convert from perl hash to database string

Depending on the type used in the database, create the appropriate string to be sent via DBI. 

=cut
sub unexpand {
	my ($self, $line) = @_;
	while (my ($key, $type) = each (%{$self->{fields}})) { 
		$line->{$key} = $self->unexpand_col ($line->{$key}, $type) if $line->{$key};
	}
}

sub unexpand_col {
	my ($self, $val, $type) = @_;
	if ($type =~ /hstore/) {
		if (ref($val) =~ /HASH/) {
			return join(',', map { qq("$_" => "$val->{$_}") } keys (%$val));
		} elsif (! ref($val)) {
			return qq("!!value" => "$val");
		}
	} elsif ($type =~ /json/) {
		return '{ ' . join(',', map { qq("$_": "$val->{$_}") } keys (%$val)) . ' }' if ref($val) =~ /HASH/;
	} elsif ($type =~ /int/) {	# hash value. Cannot be unexpanded, use it for context only
		if (ref($val) =~ /HASH/) {
			$val = join(',', map { qq("$_" => "$val->{$_}") } keys (%$val));
		}
		if ($val and $val !~ /^[\+\-]?\d+$/) {
			my @TAB = unpack('W*', $val);
			$val = 0; foreach (@TAB) { $val *= 31; $val %= (1 << 32); $val += $_; }
			$val -= (1 << 32) if $val > (1 << 31);
		}
		return $val;
	} elsif (ref($type) =~ /Composite/) {	# composite type
		if (ref($val) =~ /HASH/) {
			my %v = map { lc($_) => $val->{$_} } keys(%$val); # case-insensitive
			return sprintf('(%s)', join(',', map { $v{$_} } @{$type->{fields}})); # use names for order, but then loose them
		}
	} elsif (ref($val)) {
		return 'TABLE:' . join('&', map { qq($_=$val->{$_}) } keys (%$val)) if ref($val) =~ /HASH/;
	}	else {
		return $val;
	}
}

=head2 @array = $self->fieldList()

Returns list of field names

=cut
sub fieldList {
	keys(%{$_[0]->{fields}})
}

=head2 $sql = $self->null_or_default($fieldName)

Returns an SQL expression (i.e. to be put in a select or a where clause)

Name can be prefixed with a table name, which will be used in the return but not to find the type.

=cut
sub null_or_default {
	my ($self, $fieldName) = @_;
	my $subFieldName = $fieldName; $subFieldName =~ s/^(\w+)\.//; my $type = $self->{fields}{$subFieldName};
	if ($type !~ /\?/) { return $fieldName; }
	else {
		return "coalesce($fieldName,'')" if $type =~ /char|text/;
		return "coalesce($fieldName,0)" if $type =~ /int|num/;		
		return "coalesce($fieldName,'1900-01-01')" if $type =~ /date|stamp/;
	}
}

=head2 $ref = $self->clean(@$cols)

Remove some columns, for security reasons

=cut
sub clean ($$) {
	my ($self,$cols) = @_;
	my %copy = %$self; delete @copy{@$cols}; return bless \%copy;
}

=head2 $ref = $self->force(%$cols)

Force value of some columns, for security reasons

=cut
sub force ($$) {
	my ($self,$cols) = @_;
	my %copy = %$self; 
	while (my ($k, $v) = each(%$cols)) {
		if (ref($v) and $v->{condition}) {
			if ($copy{$k} =~ $v->{condition}) { $copy{$k} = $v->{value}; }
		} elsif (! ref($v)) {
			$copy{$k} = $v;
		}
	}
	return bless \%copy;
}



package Silvestris::Cyclotis::Database::CompositeType;

use Dancer ':syntax';
use Dancer::Plugin::Database;

our %CACHE; # cache for types

sub find {
	my ($self, $dbName, $schema, $typeName) = @_;
	my $type_contents = $CACHE{"$dbName.$schema.$typeName"};
	unless ($type_contents) {
		my $st1 = database($dbName)->prepare_cached("select attribute_name from INFORMATION_SCHEMA.ATTRIBUTES where udt_name = ? and udt_catalog = ? and udt_schema = ? order by ordinal_position");
		$st1->execute ($typeName, config->{plugins}{Database}{connections}{$dbName}{database} || config->{plugins}{Database}{database}, $schema);
		my @FIELDS1 = (); # for composite types the order is more important than the type
		while (my @t1 = $st1->fetchrow_array) { push(@FIELDS1, $t1[0]); }
		$type_contents = $CACHE{"$dbName.$schema.$typeName"} = bless ({ fields => \@FIELDS1, name => "$schema.$typeName" }, 'Silvestris::Cyclotis::Database::CompositeType');
	}
	return $type_contents;
}


1;

=head1 LICENSE

Copyright 2014-2016 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
