=head2 Format 'line'

Simplest format: "pipe separated values"

We use pipe because it is less likely to appear in a text than ';' or ',' or even tabulator.

Symbols such as pipe or new line are URL-encoded, to make the distinction with format separators.

=cut
package Silvestris::Cyclotis::Format::nodep::line;
use parent 'Silvestris::Cyclotis::Format::nodep';

=head3 Available parameters

=over 1

=item cols

Enables to decide which colums you want to display

You can either put all colums, separated by ',', use a standard URL syntax (cols=src&cols=tra...) or use $abc where a,b and c are column aliases.

Available columns :

=over 2

=item src (alias 'o' for 'origin): Source text

=item tra (alias 't'): translation text

=item author (alias 'a'): who is the first person who wrote this segment

=item changer (alias 'c'): who is the last person who wrote this segment

=item date (alias 'd'): creation date

=item changedate (alias 'u' for update): last update

=item score (alias 's'): By default, trigram matching rate. You can choose another score algorithm in the config file.

=item context (alias 'i' for 'ident'): unique string which identifies the segment. Format depends from your CAT tool

=item note (alias 'n'): a note associated to the segment

=item props (alias 'p'): properties. This is an associative array but the format depends on your CAT client.

=item row_id (alias 'r'): serial identifier for the current row (only for some glossaries)

=item src_lang (alias 'O' for 'origin): source language

=item tra_lang (alias 'T'): target language

=item mem_name (alias 'N'): name of the memory (table)

=back

The three last values, whose alias is uppercase, are meta-values: they are not in the table itself but in info about the table.
They should be identical in all lines except when you make use of inheritance.

=item separator

Use alternative sign (default to pipe '|') as separator between columns

=back

=cut		 
sub new {
	my ($class, %params) = @_;
	if (my @tab = $class->buildFieldsList($params{'line.columns'} || $params{cols}, 0)) { $params{cols} = \@tab; } else { undef $params{cols}; }
	return bless \%params, $class;
}

sub header { 
	my $self = shift; my %args = (); my @res = ();
	if ($self->{verinfo}) { my %v = $self->VERSION_INFO(); push(@res, "Cyclotis version $v{main} format-line $v{fmt}") }
	my $q = $args{query}; $q =~ s/[\r\n]/ /g; push(@res, "query: $q") if $q; # header with query, without linefeed to avoid errors	
	my $cols; $cols = join (',', @{$self->{cols}}) if $self->{cols}; push(@res, "columns: $cols") if $cols;
	if ($self->{'display-date'}) { my @t = localtime; $t[5] += 1900; $t[4]++; push(@res, "date: $t[5]-$t[4]-$t[3] $t[2]:$t[1]:$t[0]"); }
	return join("\n", map { "; $_" } @res) . "\n";
}

sub footer {
	my $self = shift; my $count = shift;
	return "; results-count: $count\n" if $count;
	return "";
}

sub produce_info { 
	my ($self, $formats, $url, $allow, $timestamp, $tables) = @_;
	$self->header(query => 'info') 
	. "default-format: $formats->{default}\n"
	. "timestamp: $timestamp\n"
	. ($tables ? "tables: $tables\n" : "\n")
	. "info-url: $url->{meta}{info}\n"
	. "struct-url: $url->{meta}{struct}\n"
	. "find-url: $url->{contents}{find}\n"
	. "save-url: $url->{contents}{save}\n"
	. "delete-url: $url->{contents}{delete}\n"
	. "allow " . ($allow->{'find-per-post'} ?  "find-per-post " : "")
	             . ($allow->{'save-per-get'} ?  "find-per-post " : "")
			   . ($allow->{'table-struct-display'} ?  "table-struct-display($allow->{'table-struct-display'}) " : "")
				 . "\n"
	. $self->footer() 
}

sub struct_info {
	my ($self, %tables) = @_;
	return $self->header('/struct-info') . ";\n"
		. join(";\n", map {
			my $tableName = $_; 
			my %options = %{$tables{$tableName}}; my $cols = $options{fields}; my $props = $options{propsstore}; delete @options{'fields','table_schema','table_name', 'propsstore'};
			"; $tableName table\n"
				. join("", map { "$_ col " 
						. (ref($cols->{$_}) =~ /Composite/ ? ($cols->{$_}{name} . '(' . join(',',@{$cols->{$_}{fields}}) . ')') : $cols->{$_}) . "\n" } keys(%$cols))
				. join("", map { "$_ opt $options{$_}\n" } grep { $_ !~ /prop|field|parent/ }  keys(%options))
				. join("", map { "$_ prop $props->{$_}\n" } keys(%$props))
		} keys (%tables))
		. "\n";
}


sub MIME { 'text/plain' }

sub produce_line { 
	my ($self,$line) = @_; 
	my @tab = $self->buildFieldsList($self->{cols}, 1);
	my %copy = map { ($_ => $self->encode($line->{$_})) } keys(%$line); 
	return join ($self->{separator} || '|', @copy{map { lc($_) } @tab}) . "\n";
}

sub encode {
	my ($self, $obj) = @_;
	if (ref ($obj) =~ /HASH/) {
		# This is an expanded json or hstore.
		# In line format, we translate it to URL-encoded
		my $res = '{'; 
		while (my ($k,$v) = each(%$obj)) { $res .= $self->encode($k) . '=' . $self->encode($v) . '&'; }
		$res =~ s/\&$/\}/ or $res .= '}'; return $res;
	} elsif (ref ($obj) =~ /ARRAY/) {
		# This is an expanded json or hstore.
		# In line format, we translate it to URL-encoded
		return '[' . join('&', map { $self->encode($_) } @$obj) . ']';
	} elsif (defined $obj) {
		$obj =~ s/[\r\n\|\=\&\%\[\]\{\}]/sprintf('%%%02x', ord($&))/eg; return $obj;
	} else {
		return ""; # must return something printable
	}
}

sub produce_error { 
	my $err = $_[1]; 
	$err = $err->code . "\t" . $err->message if ref($err) and $err->isa('Dancer::Error');
	$err =~ s/\r?\n$//; $err =~ s/\r?\n/$&!\t/g; 
	return "! $err"; 
}

sub status_line { 
	my ($self,$line) = @_; 	
	my %contents = %{$line->{contents}}; if ($line->{added}) { $contents{$_} = $line->{added}{$_} foreach grep { /date/ } keys %{$line->{added}}; }
	my $res = join ($self->{separator} || '|', $res, $self->produce_line(\%contents)); 
	if ($line->{error}) { $res = "! $line->{error}\n$res"; } 
	return $res;
}

sub show_status {
	my ($self, $level, $types, $Queries, $Lib) = @_; my $res = "";
	unless ($types !~ /cache/) {
	  	$res = "; Tables : " . scalar (keys (%Silvestris::Cyclotis::Database::Table::CACHE));
		if ($level > 1) { $res .= " (" . join(',',keys (%Silvestris::Cyclotis::Database::Table::CACHE)) . ")"; }; $res .= "\n";
		$res .= "; Queries: " . scalar(@$Queries); 
    	$res .= "(" . join(',',@$Queries) . ")" if $level > 1; 
    	$res .= "\n\n";
	}
	unless ($types !~ /libs/) {
		while (my ($lib, $hash) = each (%$Lib)) {
			$res .= "; $lib ($hash->{location}, $hash->{version}, $hash->{date})\n";
		}
	}
	return $res;
}

sub show_log {
	my $self = shift; my $log = shift;
	my $order = $log->{__ORDER__}; delete $log->{__ORDER__}; 
	my $res = ";" . join(',', @$order) . "\n";
	foreach my $key (sort keys %$log) {
		$res .= join($self->{separator} || '|', map { $log->{$key}{$_} } @$order) . "\n";
	}
	return $res;
}


sub read_line {
    shift; my $FH = shift; my $txt; ($txt, $$FH) = ($$FH =~ m!^(.+?)(?:\n|$)(.*)!);
	my @cols = split(/\|/, $txt); my %fields; foreach my $field ('src','tra','author','context','note','props') { $fields{$field} = shift(@cols); }
	if (my @tab = split(/\&/, $fields{props})) {
		$fields{props} = {}; foreach my $item (@tab) { my ($k,$v) = split(/=/,$item); $fields{props}{$k} = $v; }
	}
	if (my @tab = split(/\&/, $fields{context})) {
		$fields{context} = {}; foreach my $item (@tab) { my ($k,$v) = split(/=/,$item); $fields{context}{$k} = $v; }
	}
	return %fields if grep { defined($_) } values(%fields); return ();
}

sub needs_field { defined $_[0]->{cols} ? grep { /$_[1]/ } @{$_[0]->{cols}} : ($_[1] !~ /_/) }

1;

=head1 LICENSE

Copyright 2013-2016 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or  as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
