=head2 Format 'tmx'

Returns a Translation Memory Exchange file

=cut
package Silvestris::Cyclotis::Format::nodep::tmx;
use parent 'Silvestris::Cyclotis::Format::nodep';

=head3 Available parameters

=over 1

=item ctx2prop

Converts the context, seen as a JSON associative array, to TMX properties. 
Used by OmegaT in the project mode.

=item attr

Whenever authors and dates have to be put to "tu" or source "tuv" (srcTuv) or translation tuv (traTuv)

=back

=cut
sub new {
    my (undef, %params) = @_;
	# For OmegaT : convert context to properties
    $params{ctx2prop} = 'json' if grep { /\bc(on)?te?xt?2prop/ } keys (%params);
    return bless \%params, 'Silvestris::Cyclotis::Format::nodep::tmx';
}

sub MIME { 'application/tmx+xml' }
sub namespace_uri { 'http://www.lisa.org/tmx14' }

sub produce_error {  
	my $err = $_[1];
	if (ref($err) and $err->isa('Dancer::Error')) { $err = sprintf('<error code="%i">%s</error>', $err->code, $err->message); }
	else { $err = "<error>$err</error>"; }
<< "EOF"
<?xml version='1.0' encoding='utf-8'?>
<error>$_[1]</error>
EOF
}

sub header {
  my ($self, %args) = @_; $args{srcLang} ||= 'EN-US'; $args{adminLang} = $_[0]->{adminLang} || $args{srcLang};
  my $verInfo; if ($self->{verinfo}) { my %v = $self->VERSION_INFO(); $verInfo = qq(creationtoolversion="$v{main}") unless $self->{embedded}; }
  $query = $self->{embedded} ? '' : "<!-- $args{query} -->";
  unless ($self->{embedded}) {
	if ($self->{'display-date'}) { my @t = localtime; $t[5] += 1900; $t[4]++; $query .= "<!-- date: $t[5]-$t[4]-$t[3] $t[2]:$t[1]:$t[0] -->\n"; }
  }
  return << "EOF";
<?xml version='1.0' encoding='utf-8'?>
<!DOCTYPE tmx SYSTEM "tmx14.dtd">
<tmx version="1.4">
     <header
            o-tmf="Silvestris Cyclotis"
            creationtool="cyclotis-rest-api.pl"
            $verInfo
            segtype="sentence"
            datatype="PlainText"
            adminlang="$args{adminLang}"
            srclang="$args{srcLang}"
            />
		$query
		$verInfo
     <body>
EOF
}

sub produce_line {
    my ($self, $line) = @_;
	my %copy = %$line; foreach (values (%copy)) { s/[<>&]/sprintf('&#x%02x;', ord($&))/eg unless ref($_); }
    my %attr;
    $attr{tu} .= " tuid='$copy{context}'" if $line->{context} && (! $self->{ctx2prop});
	$attr{tu} .= " tuid='$copy{row_id}'" if $line->{row_id};
    if ($self->{attr}) {
       $attr{$self->{attr}} .= " creationid='$copy{author}' creationdate='$copy{date}'";
       $attr{$self->{attr}} .= " changeid='$copy{changer}'" if $line->{changer};
       $attr{$self->{attr}} .= " changedate='$copy{changedate}'" if $line->{changedate};
    }
	if ($self->{ctx2prop} =~ /json/i) { 
       $line->{props}{$1} = $3 while $line->{context} =~ /"(.+?)"\s*:\s*([\'\"])?(.+?)\2\s*,/;
    }
    my $propstr = "\n       ";
    if ($line->{props}) {
       while (my ($k, $v) = each (%{$line->{props}})) {
             my ($ck, $cv) = ($k, $v); $k =~ s/[<>&]/sprintf('&#x%02x;', ord($&))/eg; $v =~ s/[<>&]/sprintf('&#x%02x;', ord($&))/eg;
             $propstr .= "<prop type='$k'>$v</prop>\n       ";
       }       
    }
    $propstr .= "<note>$line->{note}</note>" if $line->{note}; # pure text
    $propstr =~ s/\n[\s\t]+$//;
    return << "EOF";
            <tu $attr{tu}>
               <prop type='Cyclotis:score'>$line->{score}</prop>
			   $propstr
               <tuv xml:lang='$self->{langSrc}' $attr{srcTuv}>
                    <seg>$copy{src}</seg>
               </tuv>
               <tuv xml:lang='$self->{langTra}' $attr{traTuv}>
                    <seg>$copy{tra}</seg>
               </tuv>
           </tu>        
EOF
}

sub footer {
    "\n\t</body>\n</tmx>\n"
}

sub status_line { 
	my ($self,$line) = @_; 
	if ($line->{error}) { $line->{contents}{props}{Insertions} = $line->{error}; } 
	else { $line->{contents}{props}{Insertions} = $line->{dbi}; }
	return $self->produce_line($line->{contents}); 
}

sub read_line {
    shift; my $FH = shift; my $txt; ($txt, $$FH) = ($$FH =~ m!^.*?(<tu.+?</tu>)(.*)!s);
	my @tuv = (); while ($txt =~ m!<tuv.+?</tuv>!gs) {  my $item = $&; push(@tuv,$1) if $item =~ m!<seg>(.+)</seg>!s; }
	my ($author) = ($txt =~ m!creationid\s*=\s*[\'\"](.+?)[\'\"]!s);
	my ($tuid) = ($txt =~ m!tuid\s*=\s*[\'\"](.+?)[\'\"]!s);
	return (src => $tuv[0], tra => $tuv[1], author => $author, context => $tuid) if $tuv[0];
	return ();
}

1;

=head1 LICENSE

Copyright 2013-2016 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or  as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
