=head1 DESCRIPTION

These are the routes to access or modify memory contents.

=cut

use Encode;
use Dancer;
use Dancer::Plugin::Database;
use Dancer::Plugin::StreamData;

use Silvestris::Cyclotis::Format;
use Silvestris::Dancer::MyRoutes;
use Silvestris::Cyclotis::Database::Table;
use Silvestris::Cyclotis::Database::Query::Search;


=head2 GET /tu/:db/:table/:cmd(.fmt)?

Does a query in the database and shows the results.

=head3 ADDITIONNAL PARAMETERS

This parameters are set as traditionnal query string parameters.

=over 1

=item version

Returns all segments valid at the given date. The table must be versionned.

=item uniq

Filter returned lines using a list of fields: if two lines have the same values for given fields, only the last one (according date) is returned

For the format of this parameter, see C<Silvestris::Cyclotis::Format::line>, parameter C<cols>.

=item sort

List of fields you want the result to be sorted with, separated by ','. 
Each field may be preceded by + or - to indicate that you want to sort ascending or descending order.

=back

=head3 Views

Optional parameter 'view' enables to restrict the result. Possible values:

=over 1

=item check

Only test if the query is correct, does not retreive the result

=item count

Only return the number of results, not the results themselves

=item page-(size)-(id)

Return exactly 'size' results or less. 

Optional id enables to pass id*size first results. Please be aware that this application is stateless,
so we cannot guarantee that the next call will return following results, as the contents of the database may have changed.

=item best-(size)

Return exactly 'size' results, the best ones regarding column 'score'.

=back

=cut
my @methods = ('get'); push (@methods,'post','put') if config->{allow}{'find-per-post'};
sv_route 'contents.find' => \@methods => '/tu/(:db/)?:table/:cmd(.:fmt)?' => sub {
	my $cmd = param('cmd'); return &doDelete() if $cmd eq 'delete' and config->{allow}{'delete-per-get'}; 
	die "Unknown find command : $cmd" unless Silvestris::Cyclotis::Database::Query::Search->sqlExpressionForCommand($cmd); # die will be kept by final handle
	var 'command' => ('contents.find.' . $cmd);
	my $fmt = Silvestris::Cyclotis::Format->for_dancer_params();
	
	my ($text) = param('query');
	my ($table) = param('table'); if (my $schema = param('schema')) { $table = "$schema.$table"; }

	my $tableRef = find Silvestris::Cyclotis::Database::Table (param('db'), $table);
	my $tags = param('tags'); unless ($tags) { $tags = config->{search}{tags} || [ '<.+?>' ]; $tags = join('|', map { "($_)" } @$tags); } $tags =~ s!\\!\\\\!;
	my @fields = grep { $fmt->needs_field($_) } qw (src_lang tra_lang mem_name);
	my $query = new Silvestris::Cyclotis::Database::Query::Search(
		tableRef => $tableRef, tableName => param('table'), schema => param('schema'),
		cmd => param('cmd'), sqlConfig => config->{sql}, tags => $tags, fields => \@fields
	);
	if ((! defined $tableRef) and ($table =~ /</)) {
		my ($tableParent, $tableChild) = split(m!<!, $table);
		$query->{tableRef} = $tableRef = find Silvestris::Cyclotis::Database::Table (param('db'), $tableChild);
		$query->{fallbackParentRef} = find Silvestris::Cyclotis::Database::Table (param('db'), $tableParent);
	}
	if (my $lang = param('lang')) { $query = $query->addParam_lang($lang, config->{dict}); }
	else {
		if (my $lang = param('langSrc')) { $query = $query->addParam_lang($lang, config->{dict}, 'Src'); }
		if (my $lang = param('langTra')) { $query = $query->addParam_lang($lang, config->{dict}, 'Tra'); }	
 	}
	if (($cmd =~ /all|version/) || (! $fmt->needs_field('score'))) { $query = $query->addParam_displayScore (1); }	# do not use score
	else { $query = $query->addParam_displayScore (param('score') || config->{search}{'displayed-score'} || 'similarity'); }

	if (my $fscore = param('filter-contents') || config->{search}{filtered}{contents}) {
		$query = $query->addParam_filterContents($fscore);
	} 
 	if (my $fscore = param('filter-input') || config->{search}{filtered}{input}) {
		$query = $query->addParam_filterInput($fscore)
			unless ($cmd =~ /all|version/);
	}
	
	if (my $uniq = param('uniq')) { $query = $query->addParam_uniq($uniq); }
	if (my $version = param('version')) { $query = $query->addParam_version($version); }	
	if (param('view') and param('view') =~ /^page\-(\d+)(?:\-(\d+))?$/) {
		my ($size, $id) = ($1, $2); $query = $query->addParam_page ($size, $id);		
	}
	if (my $sort = param('sort')) { $query = $query->addParam_sort($sort); }
	elsif (param('view') and param('view') =~ /^best-(\d+)/) { # cannot mix best view and sort
		my $size = $1; $query = $query->addParam_sort('-score')->addParam_page($size);
	}
	if (config->{debug}{sql}) { my $debug = sv_debug('sql'); &$debug("SQL = " . $query->toSQL()); }
	if (($cmd eq 'fuzzy') and (config->{search}{'optimize-fuzzy'})) {
		my $debug = config->{debug}{params}; $debug = sv_debug('params') if $debug; &$debug("Text to submit: $text") if $debug;
		my %words; $words{lc($1)}++ while $text =~ /(\w+)/gs;
		$text = join(' ', keys %words); # keep only what is used in fuzzy trigrams
		&$debug("Text submitted: $text") if $debug;
	}
	my $st = $query->execute(database, $text) or return $fmt->produce_error ($DBI::errstr);
	$text = param('query'); # change was only valid for submission
	
	my $filter_output = undef;
	if ($filter_output = param('filter-input') || config->{search}{filtered}{output}) {
		# Transform to a sub
		my @filters = split (/\-/, $filter_output);
		if ($filters[0] =~ /untag/i) { $filter_output = sub { $_ = shift; s/<.*?>//; $_ }; }	# remove all tags 
		if ($filters[0] =~ /retag/i) { # substitute by tags coming from the query text 
			$filter_output = sub {		
				my ($output, $query) = @_;
				my @queryTags = ($query =~ /<.*?>/g);
				my $i = 0; $output =~ s/<.*?>/$queryTags[$i++]/eg;
				return $output;
			}; 
		}
		if ($filters[-1] =~ /omegat/i) {	# filter only if tool_info = omegat
			my $ori = $filter_output;
			$filter_output = sub {
				my ($text, $query, $line) = @_;
				if ($line->{tool_info} && $line->{tool_info} =~ /omega\W?t/) { return &$ori($text,$query,$line); } else { return $text; }
			}
		}
		if ($filters[-1] =~ /trados/i) {	# filter only if tool_info = trados or text contains Trados-style tags
			my $ori = $filter_output;
			$filter_output = sub {
				my ($text, $query, $line) = @_;
				if ($line->{tool_info} && $line->{tool_info} =~ /trados/ || $text =~ /<\d/) { return &$ori($text,$query,$line); } else { return $text; }
			}
		}		
	}
	
	unless (server_supports_streaming && $fmt->can('produce_as_stream')) {
		return $fmt->reencode(1, $fmt->produce($st, $text, $tableRef, param('view') || '', $filter_output));
	} else {
		var 'streaming' => 1;
		content_type ($fmt->MIME() . "; charset=utf-8"); 
		return stream_data ($st, sub {
			my ($data, $writer) = @_;
			$fmt->produce_as_stream ($data, $writer, $text, $tableRef, param('view') || '', $filter_output);
			$writer->close();
			&$Silvestris::Logger::loggerFunc(); # not in hook after, which would answer before end of this function
		});		
	}
};


=head2 POST /(:db/)?:table(.fmt)?

Reads the POST contents and adds an entry to the database. 

By default, data must be in application-www-urlencoded format and contain only one line.
You can use input formats by specifying the used format as HTTP 'Content-Type' header, in which case multiple lines are supported.

Format specified at end of URL is used for the I<output>, not for the input format.

=cut
@methods = ('post','put'); push (@methods,'get') if config->{allow}{'save-per-get'};
sv_route 'contents.save' => \@methods => '/tu/(:db/)?:table' => sub {
	my ($table) = param('table'); if (my $schema = param('schema')) { $table = "$schema.$table"; }
	my $contentType = request->header('Content-Type') || 'application/x-www-form-urlencoded';
	my $outFmt = Silvestris::Cyclotis::Format->for_dancer_params(); my @lines;
	
	my $tableRef = find Silvestris::Cyclotis::Database::Table (param('db'), $table); my %fields = %{$tableRef->{fields}};
	my @fields = grep { $_ !~ /date|change/ } keys(%fields); # We never put date or changer, this is done by a Procedural/SQL rule	
	my $recheck = config->{'save-recheck'}{update}; $recheck = 'first' if $recheck and (($outFmt =~ /line/) or (ref($outFmt) =~ /line/));
	if ($contentType =~ /form-urlencoded/i) {
		my %params = params; @fields = grep { $params{$_} } @fields;
		if ($tableRef->{mem_id} && $tableRef->{std_parent}) { unshift(@fields, 'mem_id'); }
		elsif ($tableRef->{mem_path}) { unshift(@fields, 'mem_path'); }
		elsif ($tableRef->{mem_code}) { unshift(@fields, 'mem_code'); }		
		my $recheckQuery = database->prepare_cached (
			"select * from " . ($tableRef->{std_parent} || $table)
		  . ' where ' . join(' and ', map { "$_ = ?" } @fields)
		) if config->{'save-recheck'}{update};
		# Additional fields which may not be in recheck
		if (my @subProps = grep { /^prop(ertie)?s?\./i } keys(%params)) {
			if ($tableRef->{fields}{props}) {
				push(@fields, 'props'); $params{props} = {};
				foreach (@subProps) { $params{props}{substr($_,index($_,'.') + 1)} = $params{$_}; delete $params{$_}; }
			}
		}
		if (my @subTool = grep { /^tool(_info)?./i } keys(%params)) {
			if ($tableRef->{fields}{tool_info}) {
				push(@fields, 'tool_info'); $params{tool_info} = {};
				foreach (@subTool) { $params{tool_info}{substr($_,index($_,'.') + 1)} = $params{$_}; delete $params{$_}; }
			}
		}
		$tableRef->unexpand(\%params); $params{author} ||= var 'auth_user';
		my $sql = sprintf("insert into %s(date,%s) values (NOW(),%s)", ($tableRef->{std_parent} || $table), join(',',@fields), join(',', map {'?'} @fields));
		my $st = database(param('db'))->prepare_cached ($sql);
		foreach ('mem_id','mem_path','mem_code') { $params{$_} = $tableRef->{$_} if $tableRef->{$_}; }
		if ((ref ($params{src}) =~ /ARRAY/) and (ref($params{tra}) =~ /ARRAY/) and (scalar(@{$params{src}}) == scalar(@{$params{tra}}))) {
			for (my $i = 0; $i < @{$params{src}}; $i++) {
				my %values = map { ($_ => (ref($params{$_}) =~ /ARRAY/) ? $params{$_}->[$i] : $params{$_}) } @fields; 
				$tableRef->unexpand(\%values); $values{author} ||= var 'auth_user';
				my %line0 = (contents => \%values); push(@lines,\%line0); &execUpdateOneLine($st, \%line0, [@values{@fields}], $recheck, $recheckQuery); 
			}
		} else {
			my %values = map { ($_ => $params{$_}) } @fields; $tableRef->unexpand(\%values); $values{author} ||= var 'auth_user';
			my %line0 = (contents => \%values); push(@lines,\%line0); &execUpdateOneLine($st, \%line0, [@values{@fields}], $recheck, $recheckQuery); 
		}
	} else {
		if ($tableRef->{mem_id} && $tableRef->{std_parent}) { unshift(@fields, 'mem_id'); }
		elsif ($tableRef->{mem_path}) { unshift(@fields, 'mem_path'); $table = $1 if $tableRef->{mem_path} =~ m!^([\w\.]+)/!; }
		my $sql = sprintf("insert into %s(date,%s) values (NOW(),%s)", ($tableRef->{std_parent} || $table), join(',',@fields), join(',', map {'?'} @fields));
		my $st = database(param('db'))->prepare_cached ($sql);
		my $inFmt = Silvestris::Cyclotis::Format->for_mime ($contentType);
		unless ($inFmt) {
 			status 400;
 			halt $outFmt->reencode (1, $outFmt->produce_error("Bad content-type, or not supported : $contentType"));
 		}
		$inFmt = new Silvestris::Cyclotis::Format($inFmt); # convert name to object
		my $recheckQuery = database(param('db'))->prepare_cached (
			"select * from " . ($tableRef->{std_parent} || $table)
		  . ' where ' . join(' and ', map { "$_ = ?" } @fields)
		) if config->{'save-recheck'}{update};		
 		my $res = ""; my $body = request->body();
		while (my %line = $inFmt->read_line (\$body)) {
			$tableRef->unexpand(\%line); $line{mem_id} = $tableRef->{mem_id}; $line{mem_path} = $tableRef->{mem_path}; $line{author} ||= var 'auth_user'; 
			my %line0 = (contents => \%line); push(@lines,\%line0); &execUpdateOneLine($st, \%line0, [@line{@fields}], $recheck, $recheckQuery); 
		}
	}
	return $outFmt->reencode (1, $outFmt->produce_write_response(\@lines));
};

sub execUpdateOneLine {
	my ($st, $lineRef, $paramRef, $recheck, $recheckQuery) = @_;
	if (my $dbiRes = $st->execute (@$paramRef)) { 
		$lineRef->{dbi} = $dbiRes;  if ($recheck) {
			$recheckQuery->execute (@$paramRef); 
			my @recheckResults = (); while (my $h = $recheckQuery->fetchrow_hashref) { push(@recheckResults, $h); }
			$lineRef->{added} = { count => scalar(@recheckResults) };
			if ($recheck =~ /first/) { $lineRef->{added}{first} = $recheckResults[0]; }
			elsif ($recheck =~ /last/) { $lineRef->{added}{last} = $recheckResults[-1]; }
			elsif ($recheck =~ /all/) { $lineRef->{added}{list} = \@recheckResults; }
		}
	}
	else { $lineRef->{error} = $DBI::errstr; }	
}

=head2 DELETE /:table(.fmt)??context=xxx&src=yyy

Delete lines from the table. Either context or src must be provided, not necessairly both.

Note. Forbidden by default, you I<must> declare an URL to activate it.

Format specified at end of URL is used for the I<output>, not for the input format.

=cut
if (config->{url}{schemas}{contents}{delete} or config->{allow}{'delete-per-get'}) {
	sub doDelete {
		my $context = param('context'); my $src = param('src');
		my $outFmt = Silvestris::Cyclotis::Format->for_dancer_params();
		return $outFmt->produce_error("Forbidden to delete all elements in the table") unless $context or $src;

		my $recheck = config->{'save-recheck'}{delete}; $recheck = 'first' if $recheck and (($outFmt =~ /line/) or (ref($outFmt) =~ /line/));
		
		unless (ref($src) or ref($context)) {	# single line
			my %params = (); $params{src} = $src if $src; $params{context} = $context if $context; 
			my %line0 = (contents => \%params);
			if (my $dbiRes = database(param('db'))->quick_delete(param('table'), \%params)) { $line0{dbi} = $dbiRes;  } 
			else { $line0{error} = $DBI::errstr; }
			if ($recheck) {
				$line0{added}{count} = database(param('db'))->quick_count(param('table'), \%params);
				unless ($recheck =~ /count/) {
					my @list = database(param('db'))->quick_select(param('table'), \%params);
					$line0{added}{first} = $list[0] if ($recheck =~ /first/);
					$line0{added}{last} = $list[-1] if ($recheck =~ /last/);
					$line0{added}{list} = \@list if ($recheck =~ /all/);
				}
			}
			$outFmt->reencode (1, $outFmt->produce_write_response([\%line0])); 				
		} else {	# multiple lines
			my $tableRef = find Silvestris::Cyclotis::Database::Table (param('db'), param('table')); my @params; my $sql;
			if ($tableRef->{std_parent}) {
				$sql = "delete from $tableRef->{std_parent} where mem_id=?" if $tableRef->{mem_id};
				$sql = "delete from $tableRef->{std_parent} where mem_code=?" if $tableRef->{mem_code};
				push(@params, $tableRef->{mem_id} || $tableRef->{mem_code});
			}
			elsif ($tableRef->{mem_path}) {
				my $idx = length($&) - 1 if $tableRef->{mem_path} =~ m!^[\w\.]+?[\(\)/:]!;
				$sql = "delete from " . substr($tableRef->{mem_path}, 0, $idx) . " where mem_path=?";
				push(@params, $tableRef->{mem_path});
			} else {
				$sql = sprintf('delete from %s where 1=1', param('table'));
			}
			$sql .= " and context=?" if $context; $sql .= " and src=?" if $src; $sql =~ s/1=1 and//;
			push(@params, $context) if $context; push(@params,$src) if $src;
			my $st = database(param('db'))->prepare_cached ($sql); my @lines;
			my $stRecheck; if ($recheck){ $sql =~ s/^delete/select \*/; $stRecheck = database(param('db'))->prepare_cached($sql); }
			L: while (1) {
				my @params1 = map { ref($_) =~ /ARRAY/ ? (shift(@$_) || last L) : $_ } @params;
				my %line0 = (contents => { src => $src, context => $context }); push (@lines, \%line0);
				if (my $dbiRes = $st->execute (@params1)) { $line0{dbi} = $dbiRes;  }
				else { $line0{error} = $DBI::errstr; }
				if ($recheck) {
					$stRecheck->execute(@params1); my @checks = (); while (my $res = $stRecheck->fetchrow_hashref) { push(@checks,$res); }
					$line0{added}{count} = scalar(@checks);
					$line0{added}{first} = $checks[0] if ($recheck =~ /first/);
					$line0{added}{last} = $checks[-1] if ($recheck =~ /last/);
					$line0{added}{list} = \@checks if ($recheck =~ /all/);
				}
			}
			$outFmt->reencode (1, $outFmt->produce_write_response(\@lines));
		}
	}
	sv_route 'contents.delete' => delete => '' => \&doDelete;
	if (config->{allow}{'delete-per-get'}) {
		my $deleteUrl = config->{url}{schemas}{contents}{find} || '/tu/(:db/)?:table/:cmd(.:fmt)?'; $deleteUrl =~ s/:cmd/delete/;
		sv_route 'contents.delete' => get => $deleteUrl => \&doDelete;
	}	
}

dance();

=head1 LICENSE

Copyright 2013-2018 Silvestris Project (L<http://www.silvestris-lab.org/>)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
