require 'elefas'

module Elefas
    
    class ElefasWriter < ElefasClient
        def initialize(dest)
            super(dest)
        end
        
        def start_doc(docName,collection=nil)
            @conn.exec('begin transaction')
            doc_id = nil ; coll_id = nil
            if collection != nil then
                coll_id = @conn.exec("select id from EF_COLLECTION where name=?", collection, Object, 1)
                if coll_id == nil then
                    coll_id = @conn.exec("insert into EF_COLLECTION(name) values(?) returning id", collection, Object, 1)
                    puts "Collection #{collection} created (#{coll_id})"
                else    # if coll already exists also check document
                    doc_id = @conn.exec("select id from EF_DOC where name=:name and collection=:coll", { 'coll' => coll_id, 'name' => docName }, Object, 1)
                end
            else
                doc_id = @conn.exec("select id from EF_DOC where name=:name and collection is null", { 'name' => docName }, Object, 1)
            end
            if doc_id == nil then
                coll_id = 'null' if coll_id == nil
                doc_id = @conn.exec("insert into EF_DOC(name,collection) values (:file,#{coll_id}) returning id", { 'file' => docName }, Object, 1)
                @unitWriter = UnitWriter.new(@conn) unless (defined? @unitWriter and not (@unitWriter.is_a? UnitUpdater))
            else
                docCols = nil
                @conn.exec('select column_name from information_schema.columns 
                    where table_catalog=:db and table_name=:table and table_schema=:schema',
                    { 'db' => @conn.dbName, 'table' => 'ef_doc', 'schema' => 'public' }) { |row| docCols << row[1] }
                @conn.exec('update ef_doc set change_date=now() where id=' + doc_id) if docCols.any? 'change_date' 
                @unitWriter = UnitUpdater.new(@conn)  unless (defined? @unitWriter and @unitWriter.is_a? UnitUpdater)
            end
            return doc_id
        end
        
        def write(tu,doc_id)
            # tu_id is unique for a given document
            unit_id = @unitWriter.insertUnit(tu,doc_id)
            # tuv is NOT unique, even for a given language, unless if text is the same
            tu.variants.each { |lang,tuv| @unitWriter.insertSegment(tuv,lang,unit_id) }
        end  
                
        def end_doc()
            @conn.exec('commit')
        end
        
        def count_tu() @unitWriter.count_tu() end
        def count_tuv() @unitWriter.count_tuv() end
    end # class ElefasWriter
    
    class UnitWriter
        def initialize(conn)
            @unitCols = []
            conn.exec('select column_name from information_schema.columns 
                where table_catalog=:db and table_name=:table and table_schema=:schema',
                { 'db' => conn.dbName, 'table' => 'ef_unit', 'schema' => 'public' }) { |row| @unitCols << row[1] }
            @segCols = []
            conn.exec('select column_name from information_schema.columns 
                where table_catalog=:db and table_name=:table and table_schema=:schema',
                { 'db' => conn.dbName, 'table' => 'ef_seg', 'schema' => 'public' }) { |row| @segCols << row[1] }
            
            sql_units = "insert into ef_unit(tuid,doc_id) values(:tu_id,:doc_id) returning id"
            [ 'creator', 'creation_date', 'changer', 'change_date' ].each do |col|
               if @unitCols.any? col then
                   sql_units.gsub! /\) values/, ",#{col}) values"
                   sql_units.gsub! /\) returning/, ",:#{col}) returning"
               end
            end
            @st_ins_unit = conn.prepare('st_ins_unit',sql_units)
            sql_seg = 'insert into ef_seg(unit,lang,contents) values(:unit,:lang,:contents)'
            [ 'creator', 'creation_date', 'changer', 'change_date' ].each do |col|
               if @segCols.any? col then
                   sql_seg.gsub! /\) values/, ",#{col}) values"
                   sql_seg.gsub! /\)$/, ",:#{col})"
               end
            end
            @st_ins_seg = conn.prepare('st_ins_seg',sql_seg)
            @count_tu = 0; @count_tuv = 0
        end
        
        attr_reader :count_tu, :count_tuv
        
        # Insert new unit, no check needed
        def insertUnit(tu,doc_id)
            unit_id = nil
            params = { 'tu_id' => tu.id, 'doc_id' => doc_id }
            addAttributeColumns!(params,tu,@unitCols)
            @st_ins_unit.exec(params) { |row| unit_id = row["id"] }
            @count_tu = @count_tu + 1
            return unit_id
        end
        
        # Insert segment, no check needed
        def insertSegment(tuv,lang,unit_id)
            params = { 'unit' => unit_id,  'lang' => lang[0,2], 'contents' => tuv.text }
            addAttributeColumns!(params,tuv,@segCols)
            @st_ins_seg.exec(params)
            @count_tuv = @count_tuv + 1
        end
        
    private
        def addAttributeColumns!(dest,obj,cols)
            dest['creator'] = obj.creator if cols.any? 'creator'
            dest['creation_date'] = obj.creationdate if cols.any? 'creation_date'
            dest['changer'] = obj.changer if cols.any? 'changer'
            dest['change_date'] = obj.changedate if cols.any? 'change_date'            
        end
    end # class UnitWriter
        
    class UnitUpdater < UnitWriter
        def initialize(conn)
            super(conn)
            @st_sel_unit = conn.prepare('st_sel_unit',
                "select id from ef_unit where tuid = :tuid and doc_id = :doc_id")
            @st_sel_seg = conn.prepare('st_sel_seg',
                "select count(*) count from ef_seg where unit = :unit and lang = :lang and contents=:text")
        end
        
        # Insert new unit only if not exists
        def insertUnit(tu,doc_id)
            unit_id = nil
            @st_sel_unit.exec({ 'tu_id' => tu.id, 'doc_id' => doc_id }) { |row| unit_id = row["id"] }
            if unit_id != nil then return unit_id else return super(tu,doc_id) end
        end
        
        # Insert new segment unless an identical exists
        # tuv is NOT unique, even for a given language, unless if text is the same
        def insertSegment(tuv,lang,unit_id)
            exist = false
            @st_sel_seg.exec({ 'unit' => unit_id, 'lang' => lang[0,2], 'contents' => tuv.text }) { |row| exist = (row['count'].to_i > 0) }
            super(tuv,lang,unit_id) unless exist
        end        
    end # class UnitUpdater
    
    class ElefasCleaner < ElefasClient
        def initialize(dest)
            super(dest)
        end
        
        ##
        # Clean contents of the database based on a criteria
        # Parameter: table = basis of the criteria (but other tables are cleaned if they don't have segment anymore)
        # Parameter: criteria = a date interval (number + h = hours, d = days, m = months, y = years)
        def clean!(table,criteria)
            sql = " where creation_date < NOW() - "
            if criteria =~ /^\s*(\d+)\s*([hdwmyo])[a-z]*\s*$/i then
               count = $1.to_i; min = nil
               sql = sql + "'#{count}"
               case $2
               when 'h', 'H' then sql = sql + ' hours'
               when 'd', 'D' then sql = sql + ' days'
               when 'w', 'W' then sql = sql + ' weeks'
               when 'm', 'M' then sql = sql + ' months'
               when 'y', 'Y' then sql = sql + ' years'
               when 'o', 'O' then
                   table = "EF_#{table}" unless table =~ /^EF_/
                   @conn.exec("select creation_date from #{table} order by creation_date asc") do |row|
                       min = row['creation_date'] 
                       next if min == nil
                       count = count - 1
                       break if count <= 0
                   end
                   sql = " where creation_date < '#{min}'"
                   puts "Deletion before #{min}"
               end
               sql = sql + "'::interval" if sql =~ /NOW/
            end
            
            if table =~ /doc/i then
                sql = "delete from EF_DOC #{sql}"
            elsif table =~ /unit/i then
                sql = "delete from EF_UNIT #{sql}"
                sql = sql + "; delete from ef_doc where not exists(select * from ef_unit where ef_unit.doc_id = ef_doc.id)"
            elsif table =~ /seg/i then
                sql = "delete from EF_SEG #{sql}"
                sql = sql + "; delete from ef_unit where not exists(select * from ef_seg where ef_unit.id = ef_seg.unit)"
                sql = sql + "; delete from ef_doc where not exists(select * from ef_unit where ef_unit.doc_id = ef_doc.id)"
            else
                raise "Unknown or non deletable table: #{table}"
            end
            sql.split(/;/).each { |cmd| puts cmd; @conn.exec(cmd) }
        end
    end # class ElefasCleaner
    
end # module Elefas
