#! /usr/bin/ruby

if ARGV.count < 2
    puts "Syntax: #{__FILE__} DB-Name(:user:password)?(:host:port)? [command] [arguments+]"
    puts "Available commands: create-db,add-lang"
    puts "Type #{__FILE__} help [commmand] for details about one command"
    exit
end

db = ARGV.shift
cmd = ARGV.shift

if db =~ /^help/ then
    case cmd
    when /^(init|create)[\-\_]?db$/i then
         puts "Syntax: #{__FILE__} DB-Name(:user:password)?(:host:port)? #{cmd} [--owner=user]? [--langs=lang1,lang2,...]? [--attr(-table)?=(creation|change|user|date)+]"
         puts "    or: #{__FILE__} #DB-Alias #{cmd} [--creator=user]? [--langs=lang1,lang2,...]? [--attr(-table)?=(creation|change|user|date)+]"
         puts "Creates the database and empty tables"
         if ARGV.count == 0 then
             puts "Type #{__FILE__} help #{cmd} [direct|alias] for help about the two given syntaxes"
             puts "Type #{__FILE__} help #{cmd} [langs|attr] for help about the given parameters"
         else
             mode = ARGV.shift
             case mode
             when /^di/i then
                puts "\nWarning: specified user in the first parameter is the creator of the database, not the owner;"
                puts "On Postgresql < 13, must be an user with admin rights (contrarily to owner)"
                puts "If it differs from owner of the database, add the parameter --owner with desired owner"
             when /^al/i then
                puts "When using db alias, --creator is to be specified if differs from user mentioned in config file"
                puts "If password is required, say --creator=user:password"
             when /^l/i then
                puts "\nParameter --langs indicates which languages should have an optimised table+index"
                puts "This does not prevent from storing other languages but searches in the optimised ones will be faster"
                puts "By default only the global index is created, which works but is not optimal if some languages are more frequently used"
                puts "You can add an indexed language later using add-lang command, but it takes a while if the database already contained lot of segments"
             when /^at/i then
                puts "\nBy default the tables do not store author and date, unless --attr says the contrary:"
                puts "Use --attr-unit to add them in unit tables, --attr-seg for segment table(s), --attr for both"
                puts "Value = creator,creation_date,changer,change_date,or"
                puts "\tcreation = creator+creation_date"
                puts "\tchange = changer+change_date"
                puts "\tusers = creator+changer"
                puts "\tdates = creation_date+change_date"
                puts "\tall = creation+creation_date+changer+change_date"                 
             else
                puts "Unknown mode: #{mode}"
             end
         end                
    when /^(add)[\-\_]?l(an)?g$/i then
         puts "Syntax: #{__FILE__} DB-Name(:user:password)?(:host:port)? #{cmd} lang1 lang ,..."
         puts "Creates tables and indexes optimised for given language(s)"
         puts "Also moves data for this language in the new tables, which can take a while"
    when /^(add-attr)/i then
         puts "Syntax: #{__FILE__} DB-Name(:user:password)?(:host:port)? add-attr(-seg|-unit)? attributes"
         puts "Adds attributes to tables"
         puts "Value = creator,creation_date,changer,change_date,or"
         puts "\tcreation = creator+creation_date"
         puts "\tchange = changer+change_date"
         puts "\tusers = creator+changer"
         puts "\tdates = creation_date+change_date"
         puts "\tall = creation+creation_date+changer+change_date"                  
    when /^(set-quota)/i then
         puts "Syntax: #{__FILE__} DB-Name(:user:password)?(:host:port)? set-quota (doc|unit|seg) <count>"
         puts "Adds a quota trigger to avoid too big memories"
         puts "\t<count> = how many allowed entries (can use K for 1000 and M for 1000000)"
    else
        puts "Unknown command: #{cmd}"
    end
    
else    # Real actions
    
    def db_params(db)
        if db =~ /^\#/ then
            db = db.gsub(/^\#/,'')  # if comes from ENV, may not be mutable
            require 'yaml'; config = YAML.load_file "#{File.dirname(__FILE__)}/../conf/elefas.yml"        
            raise "Memory #{db} not found in config file" unless config['memories'].key? db
            config = config['memories'][db]
            if config.is_a? String then
                args = config.split(':')
            else
                args = [ config['db'], config['user'], config['pass'], config['host'], config['port'] ]
                args.collect! { |item| item.to_s }  # even port will be used as a string
            end
        else
            args = db.split(':')        
        end
        return args
    end
    
    def db_command_line(action,args)
        action = action + " --host=#{args[3]}" if args.count > 3 and args[3].length > 0 and args[3] != 'localhost'
        action = action + " --port=#{args[4]}" if args.count > 4 and args[4].length > 0 and args[4] != '5432'
        action = action + " --user=#{args[1]}" if args.count > 1 and args[1].length > 0
        if args.count > 2 and args[2] != nil and args[2].length > 0 then
            ENV['PGPASSWORD'] = args[2] ; action = action + " --no-password"
        end
        action = action + " #{args[0]}"     # db name as last parameter        
        return action
    end
    
    def sql_create_lang_table(lg,hasContents)
        res = "create table EF_SEG_#{lg}( check (lang = '#{lg.upcase}') ) inherits (EF_SEG);"
        if hasContents then
            res = res + "
                insert into EF_SEG_#{lg} select * from EF_SEG where lang = '#{lg.upcase}';
                delete from only EF_SEG where lang = '#{lg.upcase}';"
        end
        res = res + "
                create index idxFuzzy_#{lg} on EF_SEG_#{lg} USING GIST(contents gist_trgm_ops);
                create rule r_part_#{lg} as
                    on insert to EF_SEG where lang='#{lg.upcase}' 
                do instead
                    insert into EF_SEG_#{lg} values (new.*)"
        return res
    end
    
    def create_attribute_columns(action, table,cols)
        return if cols.length == 0
        sql = "alter table #{table} "
        sql = sql + "add column creator varchar(20) NULL," if cols =~ /all|creation|creator|users/
        if table =~ /doc/i then      # creation date is set during import
            sql = sql + "add column creation_date timestamp NOT NULL default NOW()," if cols =~ /all|creation|creation_date|dates/
        else
            sql = sql + "add column creation_date timestamp NULL," if cols =~ /all|creation|creation_date|dates/
        end
        sql = sql + "add column changer varchar(20) NULL," if cols =~ /all|change|changer|users/
        sql = sql + "add column change_date timestamp NULL," if cols =~ /all|change|change_date|dates/
        puts sql
        system "#{action} -c \"#{sql.gsub(/,$/,'')}\""
    end
        
    
    case cmd
    when /^(init|create)[\-\_]?db$/i then
        pgpass = nil; pgpass = ENV['PGPASSWORD'] if ENV.key? 'PGPASSWORD'
        args = db_params(db); action = db_command_line('createdb',args)
        set_owner = false; owner = ENV['USER']; owner = args[1] if args.count > 1 and args[1].length > 0  
        langs = []; attr = { 'seg' => '', 'unit' => '', doc => '' }
        ARGV.each do |item|
            if item =~ /^(--creator=(.+))$/ then  # used only when an alias is used
                args[1] = $2; args[2] = nil; args[1], args[2] = $2.split(/:/) if $2 =~ /:/
                action = db_command_line('createdb',args)
            end
            if item =~ /^(--owner=(.+))$/ then owner = $2; set_owner = true ; action.sub!(/--/, "#{$1} --") end
            if item =~ /^(--langs=(.+))$/ then $2.split(',').each { |lg| langs << lg } end
            if item =~ /^(--attr(.+)=(.+))$/ then  
                if $2 != nil then attr[$2[1..-1]] = $3 else attr['seg'] = attr['unit'] = $3 end
            end
        end
        puts action ; system action
        action.gsub!(/^createdb/, 'psql'); action.gsub!(/\s--owner=\S+/,'')
        system "#{action} -f #{File.dirname(__FILE__)}/../sql/tables.sql"
        system "#{action} -f #{File.dirname(__FILE__)}/../sql/indexes.sql"
        system "#{action} -f #{File.dirname(__FILE__)}/../sql/functions.sql"
        create_attribute_columns(action, 'ef_unit', attr['unit']) if attr['unit'].length > 0
        create_attribute_columns(action, 'ef_seg', attr['seg'])   if attr['seg'].length > 0      
        create_attribute_columns(action, 'ef_doc', attr['doc'])   if attr['doc'].length > 0      
        if set_owner then
            [ 'EF_COLLECTION', 'EF_DOC', 'EF_UNIT', 'EF_SEG' ].each do |table|
                system "#{action} -c 'alter table #{table} owner to \"#{owner}\"'"
            end
        end
        if langs.count > 0 then
           langs.each do |lg|
              lg.gsub!(/^[ix]\-/,'') ; lg = lg[0,2]   # use ISO-2 language only
              system "#{action} -c \"#{sql_create_lang_table(lg,false)}\""
              system "#{action} -c ' alter table EF_SEG_#{lg} owner to \"#{owner}\"' " if set_owner
           end
        end
        ENV['PGPASSWORD'] = pgpass if pgpass != nil
        
    when /^(add)[\-\_]?l(an)?g$/i then
        pgpass = nil; pgpass = ENV['PGPASSWORD'] if ENV.key? 'PGPASSWORD'
        action = db_command_line('psql', db_params(db))
        langs = []; ARGV.each { |lg| langs << lg.dup }
        if langs.count > 0 then
           langs.each do |lg|
              lg.gsub!(/^[ix]\-/,'') ; lg = lg[0,2]   # use ISO-2 language only
              system "#{action} -c \"#{sql_create_lang_table(lg,true)}\""
           end
        end
        ENV['PGPASSWORD'] = pgpass if pgpass != nil        
        
    when /^add-attr-(seg|unit|doc)$/i then
        pgpass = nil; pgpass = ENV['PGPASSWORD'] if ENV.key? 'PGPASSWORD'
        action = db_command_line('psql', db_params(db))
        create_attribute_columns(action, "EF_" + $1, ARGV.shift)      
        ENV['PGPASSWORD'] = pgpass if pgpass != nil        
    when /^(set-quota)/i then
        pgpass = nil; pgpass = ENV['PGPASSWORD'] if ENV.key? 'PGPASSWORD'
        action = db_command_line('psql', db_params(db))
        table = ARGV.shift; table = "EF_#{table}" if table !~ /^ef_/i
        num = ARGV.shift; num = $1.to_i * 1000 if num =~ /(\d+)\s*[Kk]/; num = $1.to_i * 1_000_000 if num =~ /(\d+)\s*[Mm]/
        puts "table = #{table} count = #{num}"
        sql = File.read("#{File.dirname(__FILE__)}/../sql/set-quota.sql")
        sql.gsub! /\%TABLE/, table; sql.gsub! /\%NUM/, num.to_s
        puts sql
        system "echo \"#{sql.gsub(/\$/, '\\$')}\" | #{action} "
        ENV['PGPASSWORD'] = pgpass if pgpass != nil                
        
    else
        puts "Unknown command: #{cmd}"
    end
end
